<?php declare(strict_types = 1);

namespace EshopCatalog\AdminModule\Components\Features;

use Core\Model\UI\BaseControl;
use Core\Model\UI\DataGrid\BaseDataGrid;
use EshopCatalog\AdminModule\Model\Features;
use EshopCatalog\AdminModule\Model\FeatureValues;
use EshopCatalog\Model\Entities\Feature;
use Nette\Application\AbortException;
use Nette\Application\BadRequestException;
use Nette\Utils\Html;

class FeatureValuesGrid extends BaseControl
{
	public ?Feature $feature = null;

	public function __construct(protected Features $featuresService, protected FeatureValues $featureValuesServices)
	{
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * ==================  Handle
	 */
	/**
	 * @throws AbortException
	 */
	public function handleDelete(string|int $id): void
	{
		$presenter = $this->presenter;
		if ($this->featureValuesServices->remove($id)) {
			$presenter->flashMessageSuccess('eshopCatalog.defaultGrid.removed');
		} else {
			$presenter->flashMessageDanger('eshopCatalog.defaultGrid.removeFailed');
		}

		if ($presenter->isAjax()) {
			$this['grid']->reload();
			$presenter->redrawControl('flashes');
		} else {
			$presenter->redirect('this');
		}
	}

	public function handleRepairPositions(): never
	{
		$this->featureValuesServices->repairPositions($this->feature->getId());
		$this->redirect('this');
	}

	public function handleMergeAllValues(): never
	{
		$this->featureValuesServices->mergeAll($this->feature->getId());
		$this->redirect('this');
	}

	public function handleSortAlphabeticallyFeatureValues(): never
	{
		$this->featureValuesServices->sortAlphabetically($this->feature->getId());
		$this->redirect('this');
	}

	public function handlePublish(array $id): void
	{
		$presenter = $this->presenter;

		if ($this->featureValuesServices->setPublishMultiple($id, 1)) {
			$presenter->flashMessageSuccess('default.published');
		} else {
			$presenter->flashMessageDanger('default.error');
		}

		$this['grid']->reload();
		$presenter->redrawControl('flashes');
	}

	public function handleUnPublish(array $id): void
	{
		$presenter = $this->presenter;

		if ($this->featureValuesServices->setPublishMultiple($id, 0)) {
			$presenter->flashMessageSuccess('default.unPublished');
		} else {
			$presenter->flashMessageDanger('default.error');
		}

		$this['grid']->reload();
		$presenter->redrawControl('flashes');
	}

	/*******************************************************************************************************************
	 * ==================  Components
	 */

	protected function createComponentGrid(): BaseDataGrid
	{
		$grid = $this->createGrid();
		$grid->setDefaultPerPage(50);
		$grid->setRememberState();

		if (!$this->feature) {
			return $grid;
		}

		$qb = $this->featureValuesServices->getEr()->createQueryBuilder('fv')->addSelect('fvt')
			->join('fv.featureValueTexts', 'fvt', 'WITH', 'fvt.lang = :lang')
			->setParameter('lang', $this->translator->getLocale())
			->andWhere('fv.feature = :id')->setParameter('id', $this->feature->getId())
			->orderBy('fv.position');
		$grid->setDataSource($qb);

		$grid->setSortable();
		$grid->setSortableHandler('featureValuesGrid:gridSortableRow!');

		// Columns
		$grid->addColumnText('name', 'eshopCatalog.defaultGrid.value')->setRenderer(fn($row,
		) => Html::el('a', ['href' => $this->presenter->link('VariantsFeatures:editFeatureValue', $row->getId())])
			->setText($row->getFeatureValueText($this->translator->getLocale())->name));
		$grid->addColumnStatus('isPublished', 'default.isPublished')
			->setAlign('center')
			->addOption(1, 'default.publish')
			->setIcon('check')
			->setClass('btn-success')
			->setShowTitle(false)
			->endOption()
			->addOption(0, 'default.unPublish')
			->setIcon('times')
			->setClass('btn-danger')
			->setShowTitle(false)
			->endOption()
			->onChange[] = $this->gridPublishChange(...);

		// Filters
		$grid->addFilterText('name', '', 'fvt.name');
		$grid->addFilterSelect('isPublished', '', [
			''  => '',
			'1' => $this->t('default.yes'),
			'0' => $this->t('default.no'),
		]);

		// Actions
		$grid->addAction('edit', '', 'VariantsFeatures:editFeatureValue')->setIcon('edit')->setBsType('primary');
		$grid->addAction('delete', '', 'delete!')
			->setIcon('times')
			->setBsType('danger')
			->addClass('ajax')
			->setConfirm('default.reallyDelete');

		$grid->addGroupAction('default.publish')->onSelect[]   = $this->handlePublish(...);
		$grid->addGroupAction('default.unPublish')->onSelect[] = $this->handleUnPublish(...);

		return $grid;
	}

	/*******************************************************************************************************************
	 * =================  Grid function
	 */

	public function gridPublishChange(int $id, int $newStatus): void
	{
		$presenter = $this->presenter;

		if ($this->featureValuesServices->setPublish($id, $newStatus)) {
			$presenter->flashMessageSuccess('default.publishChanged');
		} else {
			$presenter->flashMessageDanger('default.publishChangeFailed');
		}

		if ($presenter->isAjax()) {
			$this['grid']->redrawItem($id);
			$presenter->redrawControl('flashes');
		} else {
			$presenter->redirect('this');
		}
	}

	public function handleGridSortableRow(): void
	{
		$presenter = $this->presenter;
		$request   = $presenter->getHttpRequest();
		$id        = $request->getPost('id');
		$position  = $request->getPost('position');

		if ($id != null && $position != null && $this->featureValuesServices->setPosition($id, $position)) {
			$presenter->flashMessageSuccess('default.positionChanged');
		} else {
			$presenter->flashMessageDanger('default.positionChangeFailed');
		}

		if ($presenter->isAjax()) {
			$presenter->redrawControl('flashes');
			$this['grid']->reload();
		} else {
			$this->redirect('this');
		}
	}

	/*******************************************************************************************************************
	 * ==================  GET / SET
	 */
	/**
	 * @throws BadRequestException
	 */
	public function setFeature(string|int $id): void
	{
		$this->feature = $this->featuresService->get($id);

		if (!$this->feature) {
			$this->presenter->error();
		}
	}

}
