<?php declare(strict_types = 1);

namespace EshopCatalog\AdminModule\Components\ProductAssign;

use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use EshopCatalog\AdminModule\Model\Products;
use EshopCatalog\AdminModule\Model\ProductsAssignedToCustomers;
use EshopCatalog\Model\Entities\Product;
use EshopCatalog\Model\Entities\ProductAssignCustomer;
use EshopOrders\AdminModule\Model\Customers;
use Exception;
use Nette\Application\LinkGenerator;
use Nette\Utils\ArrayHash;

class ProductsToCustomersForm extends BaseControl
{
	public function __construct(
		protected ProductsAssignedToCustomers $productsAssignedToCustomers,
		protected Products                    $products,
		protected Customers                   $customers,
		protected LinkGenerator               $linkGenerator,
	)
	{
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * ======================== Components
	 */

	protected function createComponentForm(): BaseForm
	{
		$form = $this->createForm();
		$form->setAjax();
		$form->setShowLangSwitcher(false);
		$form->getElementPrototype()
			->setAttribute('id', 'eshopCatalog_productsToCustomersForm');
		$form->addHidden('productId');
		$form->addHidden('customerId');

		$form->addBool('hideProductInListing', 'eshopCatalog.productAssignCustomer.hideProductInListing')
			->setDescription('eshopCatalog.productAssignCustomer.hideProductInListingDescription')
			->setDefaultValue(1);

		$form->addText('product', 'eshopCatalog.product.searchProduct')
			->setHtmlAttribute('data-autocomplete-name', 'productsSetAsVariantsFor')
			->setHtmlAttribute('data-autocomplete-target', $form->getComponent('productId')->getHtmlId())
			->setHtmlAttribute('data-autocomplete-keys', 'id,code1,ean,name')
			->setHtmlAttribute(
				'data-autocomplete-url',
				$this->linkGenerator->link('EshopCatalog:Cron:Products:loadAll'),
			);

		$form->addText('customer', 'eshopOrders.customer.searchCustomer')
			->setHtmlAttribute('data-autocomplete-name', 'productsSetAsVariantsFor')
			->setHtmlAttribute('data-autocomplete-target', $form->getComponent('customerId')->getHtmlId())
			->setHtmlAttribute('data-autocomplete-keys', 'id,name')
			->setHtmlAttribute(
				'data-autocomplete-url',
				$this->linkGenerator->link('EshopOrders:Admin:customers:jsonList'),
			);

		$form->addSaveCancelControl();
		$form->getComponent('saveControl')->removeComponent($form->getComponent('saveControl')['save']);

		$form->onValidate[] = $this->formValidate(...);
		$form->onSuccess[]  = $this->formSuccess(...);

		return $form;
	}

	public function formValidate(BaseForm $form, ArrayHash $values): void
	{
		if (!$values->productId || !$this->products->get($values->productId)) {
			$form->getComponent('product')->addError('eshopCatalog.productAssignCustomer.missingProduct');
		}
		if (!$values->customerId || !$this->customers->get($values->customerId)) {
			$form->getComponent('customer')->addError('eshopCatalog.productAssignCustomer.missingCustomer');
		}

		if ($form->hasErrors()) {
			$this->redrawControl('form');
		}
	}

	public function formSuccess(BaseForm $form, ArrayHash $values): bool
	{
		try {
			/** @var Product $product */
			$product = $this->products->get($values->productId);
			$entity  = new ProductAssignCustomer(
				$product,
				$this->customers->getReference($values->customerId),
			);

			$product->disableListing = (int) $values->hideProductInListing;

			$this->em->persist($product);
			$this->em->persist($entity);

			$this->em->flush();
			$this->presenter->flashMessageSuccess('default.saved');
		} catch (Exception $e) {
			$form->addError($e->getMessage());
			$this->redrawControl('form');

			return false;
		}

		return true;
	}

}
