<?php declare(strict_types = 1);

namespace EshopCatalog\AdminModule\Components\Products;

use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use Doctrine\ORM\Query\Expr\Join;
use EshopCatalog\AdminModule\Model\Features;
use EshopCatalog\AdminModule\Model\FeatureValues;
use EshopCatalog\AdminModule\Model\Products;
use EshopCatalog\FrontModule\Model\CacheService;
use EshopCatalog\Model\Entities\FeatureProduct;
use EshopCatalog\Model\Entities\Product;
use EshopCatalog\Model\Helpers\VariantsHelper;
use Exception;
use Nette\Caching\Cache;
use Nette\Http\Request;
use Nette\Utils\ArrayHash;
use Nette\Utils\Json;

class ProductsFeatureForm extends BaseControl
{
	public array    $products = [];
	private ?string $ids      = '';

	protected ?array $features = null;

	public function __construct(
		protected Products       $productServices,
		protected Features       $featuresService,
		protected FeatureValues  $featureValuesService,
		protected VariantsHelper $variantsHelper,
		protected CacheService   $cacheService,
		protected Request        $request,
	)
	{
	}

	public function render(): void
	{
		$this->template->featuresJson = Json::encode($this->getFeatures());
		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * ========================  COMPONENTS
	 */

	protected function createComponentForm(): BaseForm
	{
		$form = $this->createForm();

		$form->addHidden('produtcIds', $this->ids);

		$form->addSelect('feature', 'eshopCatalog.productFeature.feature', $this->getFeatures()['features']);
		$form->addSelect(
			'value',
			'eshopCatalog.productFeature.value',
			$this->getFeatures()['values'][key($this->getFeatures()['features'])] ?? [],
		);
		$form->addCheckbox('keepSameFeatures', 'eshopCatalog.productFeature.keepSameFeatures');
		$form->addCheckbox('applyToVariants', 'eshopCatalog.productFeature.applyToVariants');

		$form->addSaveCancelControl();

		// TODO deprecated?
		$feature = $this->request->getPost('feature');
		if ($feature) {
			$form->getComponent('value')->setItems($this->getFeatures()['values'][$feature]);
		}

		$form->getComponent('saveControl')['save']->setValidationScope(null);
		$form->getComponent('saveControl')['saveAndClose']->setValidationScope(null);

		$form->onSuccess[] = $this->formSuccess(...);

		return $form;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values): bool
	{
		try {
			$featureId = $values->feature;
			$valueId   = $this->presenter->getHttpRequest()->getPost('value');

			$ids    = [];
			$exists = [];

			foreach ($this->products as $prod) {
				$ids[]                  = $prod->getId();
				$exists[$prod->getId()] = false;
			}

			if ($values->applyToVariants) {
				$ids = array_merge($ids, ...$this->variantsHelper->getProductVariants($ids));
			}

			if ($values->keepSameFeatures) {
				foreach ($this->em->getRepository(FeatureProduct::class)->createQueryBuilder('fp')
					         ->select('fp.id, IDENTITY(fp.product) as prod, IDENTITY(fp.featureValue) as val')
					         ->where('fp.product IN (:ids)')
					         ->andWhere('fp.featureValue = :val')
					         ->setParameters([
						         'ids' => $ids,
						         'val' => $valueId,
					         ])->getQuery()->getArrayResult() as $row) {
					$exists[$row['prod']] = $row['id'];
				}
			} else {
				foreach ($this->em->getRepository(FeatureProduct::class)->createQueryBuilder('fp')
					         ->select('fp.id, IDENTITY(fp.product) as prod, IDENTITY(fp.featureValue) as val')
					         ->innerJoin('fp.featureValue', 'fv', Join::WITH, 'fv.feature = :featureId')
					         ->where('fp.product IN (:ids)')
					         ->setParameters([
						         'ids'       => $ids,
						         'featureId' => $featureId,
					         ])->getQuery()->getArrayResult() as $row) {
					$exists[$row['prod']] = $row['id'];
				}
			}

			foreach ($ids as $id) {
				$exist = $exists[$id] ?? false;

				/** @var FeatureProduct $featureProduct */
				if ($exist === false) {
					$featureProduct = new FeatureProduct(
						$this->productServices->getReference($id),
						$this->featureValuesService->getReference($valueId)
					);
				} else {
					/** @var FeatureProduct $featureProduct */
					$featureProduct = $this->em->getReference(FeatureProduct::class, $exist);
					$featureProduct->setFeatureValue($this->featureValuesService->getReference((int) $valueId));
				}

				$this->em->persist($featureProduct);
			}

			$this->em->flush();
			$form->addCustomData('productIds', $values->produtcIds);

			$this->cacheService->clean('product', [
				Cache::Tags => array_map(static fn($id,
				) => 'product/' . $id, explode('-', (string) $values->produtcIds)),
			]);

			$this->presenter->flashMessageSuccess('eshopCatalog.productForm.productSaved');
		} catch (Exception $e) {
			$form->addError($e->getMessage());

			return false;
		}

		return true;
	}

	/*******************************************************************************************************************
	 * ========================  GET / SET
	 */

	public function setProducts(string $ids): void
	{
		$this->ids = $ids;
		/** @var string[] $productIds */
		$productIds = explode('-', $ids);
		foreach ($productIds as $id) {
			$this->products[$id] = $this->em->getRepository(Product::class)->find($id);
		}

		if (count($this->products) === 0) {
			$this->presenter->error();
		}
	}

	protected function getFeatures(): array
	{
		if ($this->features === null) {
			$features = [];
			foreach ($this->featuresService->getEr()->createQueryBuilder('f')->select('f.id, ft.name')
				         ->join('f.featureTexts', 'ft', 'WITH', 'ft.lang = :lang')
				         ->setParameter('lang', $this->translator->getLocale())
				         ->where('f.isPublished = 1')
				         ->getQuery()->getResult() as $v) {
				$features[$v['id']] = $v['name'];
			}

			$values = [];
			foreach ($this->featureValuesService->getEr()
				         ->createQueryBuilder('fv')
				         ->select('fv.id, fvt.name, IDENTITY(fv.feature) as featureId')
				         ->join('fv.featureValueTexts', 'fvt', 'WITH', 'fvt.lang = :lang')
				         ->setParameter('lang', $this->translator->getLocale())
				         ->where('fv.isPublished = 1')
				         ->getQuery()
				         ->getResult() as $v) {
				$values[$v['featureId']][$v['id']] = $v['name'];
			}

			$this->features = ['features' => $features, 'values' => $values];
		}

		return $this->features;
	}

}
