<?php declare(strict_types = 1);

namespace EshopCatalog\AdminModule\Components\Related;

use Core\Model\UI\BaseControl;
use Core\Model\UI\DataGrid\BaseDataGrid;
use Doctrine\ORM\Query\Expr\Join;
use EshopCatalog\AdminModule\Model\Repository\RelatedGroups;
use EshopCatalog\Model\Entities\RelatedGroup;
use Nette\Application\AbortException;

class RelatedGroupsGrid extends BaseControl
{
	public function __construct(protected RelatedGroups $relatedGroups, protected IRelatedGroupFormFactory $formFactory)
	{
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * ==================  Handle
	 */

	public function handleAdd(): void
	{
		$this->template->modalTitle = $this->t('eshopCatalog.title.newRelatedGroup');
		$this->template->modal      = 'form';
		$this->redrawControl('modal');
	}

	public function handleEdit(int $id): void
	{
		$this['form']->id = $id;
		$this['form']->setRelatedGroup($id);
		$this->template->modalTitle = $this->t(
			'eshopCatalog.title.editRelatedGroup',
			['title' => $this['form']->relatedGroup->getText()->getName()],
		);
		$this->template->modal      = 'form';
		$this->redrawControl('modal');
	}

	public function handleDelete(int $id): void
	{
		$presenter = $this->presenter;
		if ($this->relatedGroups->remove($id)) {
			$presenter->flashMessageSuccess('default.removed');
		} else {
			$presenter->flashMessageDanger('default.removeFailed');
		}

		$this['grid']->reload();
		$presenter->redrawControl('flashes');
	}

	/*******************************************************************************************************************
	 * ==================  Components
	 */

	protected function createComponentGrid(): BaseDataGrid
	{
		$grid = $this->createGrid();
		$grid->setDefaultPerPage(50);

		$qb = $this->relatedGroups->getEr()->createQueryBuilder('rg')
			->addSelect('rgt')
			->leftJoin('rg.texts', 'rgt', Join::WITH, 'rgt.lang = :lang')
			->setParameter('lang', $this->translator->getLocale())
			->orderBy('rg.position');

		$grid->setDataSource($qb);

		$grid->setSortable();
		$grid->setSortableHandler('relatedGroupsGrid:gridSortableRow!');

		// Columns
		$grid->addColumnText('name', 'eshopCatalog.relatedGroup.name')->setRenderer(fn(RelatedGroup $row,
		) => $row->getText($this->translator->getLocale())->getName());
		$grid->addColumnStatus('isPublished', 'eshopCatalog.relatedGroup.isPublished')
			->setAlign('center')
			->addOption(1, 'default.publish')
			->setIcon('check')
			->setClass('btn-success')
			->setShowTitle(false)
			->endOption()
			->addOption(0, 'default.unPublish')
			->setIcon('times')
			->setClass('btn-danger')
			->setShowTitle(false)
			->endOption()
			->onChange[] = $this->gridPublishChange(...);

		// Actions
		$grid->addAction('edit', '', 'edit!')->setIcon('edit')->setBsType('primary')
			->addClass('ajax');
		$grid->addAction('delete', '', 'delete!')->setIcon('times')->setBsType('danger')
			->addClass('ajax')
			->setConfirm('default.reallyDelete');

		return $grid;
	}

	protected function createComponentForm(): RelatedGroupForm
	{
		$control = $this->formFactory->create();
		if ($this->getParameter('id')) {
			$control->setRelatedGroup((int) $this->getParameter('id'));
		}

		$control['form']->onSuccessSave[]         = function() {
			$this->presenter->redrawControl('flashes');
			$this['grid']->reload();
		};
		$control['form']->onSuccessSaveAndClose[] = function() {
			$this->presenter->payload->hideModal = true;
			$this->presenter->redrawControl('flashes');
			$this['grid']->reload();
		};
		$control['form']->getComponent('saveControl')->closeModalOnCancel();

		return $control;
	}

	/*******************************************************************************************************************
	 * =================  Grid function
	 */
	/**
	 * @param int $id
	 *
	 * @throws AbortException
	 */
	public function gridPublishChange($id, int|bool $newStatus): void
	{
		$presenter = $this->presenter;

		if ($this->relatedGroups->setPublish($id, $newStatus)) {
			$presenter->flashMessageSuccess('default.publishChanged');
		} else {
			$presenter->flashMessageDanger('default.publishChangeFailed');
		}

		$this['grid']->redrawItem($id);
		$presenter->redrawControl('flashes');
	}

	public function handleGridSortableRow(): void
	{
		$presenter = $this->presenter;
		$request   = $presenter->getHttpRequest();
		$id        = $request->getPost('id');
		$position  = $request->getPost('position');

		if ($id != null && $position != null && $this->relatedGroups->setPosition((int) $id, (int) $position)) {
			$presenter->flashMessageSuccess('default.positionChanged');
		} else {
			$presenter->flashMessageDanger('default.positionChangeFailed');
		}

		$presenter->redrawControl('flashes');
		$this['grid']->reload();
	}

}
