<?php declare(strict_types = 1);

namespace EshopCatalog\AdminModule\Components\Seller;

use Core\AdminModule\Model\Sites;
use Core\Model\Countries;
use Core\Model\Helpers\FormHelper;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use Currency\AdminModule\Model\Currencies;
use Currency\Model\Config as CurrencyConfig;
use EshopCatalog\AdminModule\Model\Event\SellerEvent;
use EshopCatalog\AdminModule\Model\Event\SellerFormEvent;
use EshopCatalog\AdminModule\Model\Event\SetSellerFormDataEvent;
use EshopCatalog\AdminModule\Model\Sellers;
use EshopCatalog\Model\Entities\Seller;
use EshopCatalog\Model\Entities\SellerBankAccount;
use EshopCatalog\Model\Entities\SellerInSite;
use Exception;
use Nette\Application\IPresenter;
use Nette\Caching\Cache;
use Nette\InvalidArgumentException;
use Nette\Utils\ArrayHash;

class SellerForm extends BaseControl
{
	/** @var int|null @persistent */
	public ?int       $id     = null;
	protected ?Seller $seller = null;

	public function __construct(
		protected Sellers    $sellersService,
		protected Sites      $sitesService,
		protected Countries  $countries,
		protected Currencies $currencies,
	)
	{
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * ==================  Components
	 */

	protected function createComponentForm(): BaseForm
	{
		$form = $this->createForm();
		$form->setAjax();
		$form->setShowLangSwitcher(false);

		$sites       = $this->sitesService->getOptionsForSelect(false);
		$sitesInUse  = $this->sellersService->getSitesInUse();
		$sellerSites = [];

		if ($this->seller) {
			foreach ($this->seller->getSites() as $v) {
				$sellerSites[$v->getSite()->getIdent()] = $v;
			}
		}

		foreach ($sitesInUse as $k => $v) {
			if (isset($sellerSites[$k])) {
				unset($sitesInUse[$k]);
			} else {
				$sites[$k] .= ' (' . $v['sellerName'] . ')';
			}
		}

		$form->addText('title', 'eshopCatalog.seller.title')
			->setMaxLength(255)
			->setRequired();
		$form->addText('name', 'eshopCatalog.seller.name')
			->setMaxLength(255)
			->setRequired();
		$form->addText('ic', 'eshopCatalog.seller.ic')
			->setMaxLength(12)
			->setNullable();
		$form->addText('dic', 'eshopCatalog.seller.dic')
			->setMaxLength(14)
			->setNullable();
		$form->addText('email', 'eshopCatalog.seller.email')
			->setMaxLength(255)
			->setRequired();
		$form->addText('phone', 'eshopCatalog.seller.phone')
			->setMaxLength(255);
		$form->addBool('sendOrder', 'eshopCatalog.seller.sendOrder')
			->setDefaultValue(1);
		$form->addBool('sendStatus', 'eshopCatalog.seller.sendStatus')
			->setDefaultValue(1);
		$form->addSelect('sendEmailType', 'eshopCatalog.seller.sendEmailType', [
			Seller::EMAIL_TYPE_SELLER => 'eshopCatalog.seller.sendEmailTypeList.seller',
			Seller::EMAIL_TYPE_SITE   => 'eshopCatalog.seller.sendEmailTypeList.site',
		]);

		$form->addCheckboxList('sites', 'eshopCatalog.seller.eshops', $sites)
			->setDisabled(array_keys($sitesInUse));

		foreach (['', '2'] as $k) {
			$container = $form->addContainer('address' . $k);
			$container->addText('address', 'eshopCatalog.seller.street')
				->setMaxLength(255)
				->setNullable();
			$container->addText('city', 'eshopCatalog.seller.city')
				->setMaxLength(255)
				->setNullable();
			$container->addText('postal', 'eshopCatalog.seller.postal')
				->setMaxLength(255)
				->setNullable();
			$container->addSelect('country', 'eshopCatalog.seller.country', $this->countries->getAllNameColumn());
		}

		$container = $form->addContainer('bankAccount');
		foreach (CurrencyConfig::load('whitelist') as $curr) {
			$currContainer = $container->addContainerWithLabel(strtolower((string) $curr), $curr);

			$currContainer->addText('bankName', 'eshopCatalog.sellerBankAccount.bankName')
				->setMaxLength(255)
				->setNullable();
			$currContainer->addText('numberPart1', 'eshopCatalog.sellerBankAccount.numberPart1')
				->setMaxLength(255)
				->setNullable();
			$currContainer->addText('numberPart2', 'eshopCatalog.sellerBankAccount.numberPart2')
				->setMaxLength(255)
				->setNullable();
			$currContainer->addText('swift', 'eshopCatalog.sellerBankAccount.swift')
				->setMaxLength(255)
				->setNullable();
			$currContainer->addText('iban', 'eshopCatalog.sellerBankAccount.iban')
				->setMaxLength(255)
				->setNullable();
			$currContainer->addText('note', 'eshopCatalog.sellerBankAccount.note')
				->setMaxLength(255)
				->setNullable();
		}

		$form->addContainer('extendedContainers');

		$form->addSaveCancelControl();

		$form->onSuccess[] = $this->formSuccess(...);

		$this->monitor(IPresenter::class, function() use ($form) {
			$sellerFormEvent           = new SellerFormEvent($form);
			$sellerFormEvent->template = $this->template;
			$this->eventDispatcher->dispatch($sellerFormEvent, 'eshopCatalog.createSellerForm');
		});

		return $form;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values): bool
	{
		try {
			/** @var Seller $seller */
			$seller       = $this->id ? $this->sellersService->get($this->id) : new Seller(
				$values->title, $values->name
			);
			$sites        = $seller->getSites()->toArray();
			$bankAccounts = $seller->getBankAccounts()->toArray();

			$tmp = (array) $values;
			unset($tmp['address'], $tmp['address2']);
			FormHelper::fillEntityByValues($seller, $tmp);

			foreach (array_diff(array_keys($sites), $values->sites) as $v) {
				$this->em->remove($sites[$v]);
			}

			foreach (array_diff($values->sites, array_keys($sites)) as $v) {
				$inSite = new SellerInSite($seller, $this->sitesService->getReference($v));
				$this->em->persist($inSite);
			}

			foreach (['', 2] as $k) {
				foreach (['address', 'city', 'postal', 'country'] as $v) {
					$seller->{$v . $k} = $values->{'address' . $k}->$v;
				}
			}
			$seller->setIc($values->ic ?: null);
			$seller->setDic($values->dic ?: null);

			$this->em->persist($seller);

			foreach ($values->bankAccount as $code => $data) {
				$bankAccount               = $bankAccounts[$code] ?? new SellerBankAccount($seller);
				$bankAccount->currencyCode = $code;
				FormHelper::fillEntityByValues($bankAccount, $data);
				$this->em->persist($bankAccount);
			}

			$this->eventDispatcher->dispatch(new SellerEvent($seller, $values), 'eshopCatalog.sellerBeforeSave');

			$this->em->persist($seller)->flush();
			$form->addCustomData('sellerId', $seller->getId());
			$this->presenter->flashMessageSuccess('default.saved');

			$cache = new Cache($this->cacheStorage, \EshopCatalog\FrontModule\Model\Sellers::CACHE_NAMESPACE);
			$cache->clean([Cache::Tags => [\EshopCatalog\FrontModule\Model\Sellers::CACHE_NAMESPACE]]);
		} catch (Exception $e) {
			$form->addError($e->getMessage());
			$this->redrawControl('form');

			return false;
		}

		return true;
	}

	public function setSeller(int $id): void
	{
		$this->id     = $id;
		$this->seller = $seller = $this->sellersService->get($id);

		if (!$seller) {
			throw new InvalidArgumentException;
		}

		FormHelper::fillContainerByEntity($this['form'], $seller);
		foreach ($seller->getBankAccounts()->toArray() as $code => $account) {
			if (isset($this['form']['bankAccount'][$code])) {
				FormHelper::fillContainerByEntity($this['form']->getComponent('bankAccount')[$code], $account);
			}
		}

		$form = $this['form'];
		$d    = [
			'sites' => $seller->getSitesIdent(),
		];

		$this->eventDispatcher->dispatch(new SetSellerFormDataEvent($form, $seller), 'eshopCatalog.setSellerFormData');

		foreach (['', 2] as $k) {
			foreach (['address', 'city', 'postal'] as $v) {
				$d['address' . $k][$v] = $seller->{$v . $k};
			}

			if ($seller->{'country' . $k}) {
				$country = $this->countries->get($seller->{'country' . $k});
				if ($country) {
					$d['address' . $k]['country'] = $country->getId();
				}
			}
		}

		$form->setDefaults($d);
	}

	public function getSeller(): ?Seller { return $this->seller; }

}
