<?php declare(strict_types = 1);

namespace EshopCatalog\AdminModule\Model;

use Contributte\Translation\Translator;
use Core\Model\Helpers\BaseEntityService;
use Core\Model\Helpers\Strings;
use Core\Model\Helpers\Traits\TPosition;
use Core\Model\Helpers\Traits\TPublish;
use Doctrine\ORM\Mapping\ClassMetadataInfo;
use Doctrine\ORM\Query\Expr\Join;
use EshopCatalog\FrontModule\Model\CacheService;
use EshopCatalog\Model\Entities\Feature;
use Exception;
use Nette\Caching\Cache;

/**
 * @method Feature|null getReference($id)
 * @method Feature[] getAll()
 * @method Feature|null get($id)
 */
class Features extends BaseEntityService
{
	use TPublish;
	use TPosition;

	protected $entityClass = Feature::class;

	public function __construct(
		protected CacheService  $cacheService,
		protected Translator    $translator,
		protected FeatureValues $featureValues,
	)
	{
	}

	/**
	 * @param int $id
	 *
	 * @throws Exception
	 */
	public function setUseAsFilter($id, int|bool $state): bool
	{
		if ($item = $this->get($id)) {
			$item->useAsFilter = $state;
			$this->em->persist($item)->flush();

			return true;
		}

		return false;
	}

	/**
	 * @param int $id
	 *
	 * @throws Exception
	 */
	public function setPosition($id, int|string $position): bool
	{
		if ($item = $this->getReference($id)) {
			$item->setPosition((int) $position);
			$this->em->persist($item);
			$this->em->flush();

			return true;
		}

		return false;
	}

	public function mergeAll(): void
	{
		$delete = [];
		$this->em->getConnection()->beginTransaction();
		try {
			foreach ($this->getEr()
				         ->createQueryBuilder('f')
				         ->select('f.id')
				         ->leftJoin('f.featureTexts', 'ft')
				         ->where('ft IS NULL')
				         ->getQuery()
				         ->getArrayResult() as $row) {
				$delete[] = $row['id'];
			}

			foreach (array_chunk($delete, 200) as $chunk) {
				$this->em->createQuery('DELETE FROM ' . $this->entityClass . ' f WHERE f.id IN (:id)')
					->setParameter('id', $chunk)->execute();
			}
			$delete = null;

			$fks = [];
			foreach ($this->em->getConnection()->fetchAllAssociative(
				"
				SELECT *
					FROM information_schema.KEY_COLUMN_USAGE
					WHERE
  						REFERENCED_TABLE_NAME = 'eshop_catalog__feature'
  						AND REFERENCED_COLUMN_NAME = 'id'
  						AND CONSTRAINT_SCHEMA = '{$this->em->getConnection()->getDatabase()}'",
			) as $tmp) {
				if ($tmp['TABLE_NAME'] == 'eshop_catalog__feature_texts') {
					continue;
				}

				$fks[] = $tmp;
				$tmp   = null;
			}

			foreach ($this->translator->getLocalesWhitelist() as $l) {
				foreach ($this->getEr()->createQueryBuilder('f')->select('ft.name, count(f.id)')
					         ->join('f.featureTexts', 'ft', 'WITH', 'ft.lang = :lang')
					         ->setParameter('lang', $l)
					         ->groupBy('ft.name')->having('count(f.id) > 1')
					         ->getQuery()->getArrayResult() as $duplicate) {
					$ids = [];

					foreach ($this->getEr()
						         ->createQueryBuilder('f')
						         ->select('f.id')
						         ->where('ft.name = :name')
						         ->setParameter('name', $duplicate['name'])
						         ->join('f.featureTexts', 'ft', 'WITH', 'ft.lang = :lang')
						         ->setParameter('lang', $l)
						         ->getQuery()
						         ->getArrayResult() as $tmp) {
						$ids[] = $tmp['id'];
					}

					$first = array_shift($ids);

					$this->em->getConnection()
						->executeStatement(
							"UPDATE eshop_catalog__feature_value SET feature_id = $first WHERE feature_id IN (" . implode(
								',',
								$ids,
							) . ")",
						);

					foreach ($fks as $fk) {
						$this->em->getConnection()
							->executeStatement(
								"UPDATE {$fk['TABLE_NAME']} SET {$fk['COLUMN_NAME']} = $first WHERE {$fk['COLUMN_NAME']} IN (" . implode(
									',',
									$ids,
								) . ")",
							);
					}

					$this->em->createQuery('DELETE FROM ' . $this->entityClass . ' f WHERE f.id IN(:id)')
						->setParameter('id', $ids)
						->execute();
				}
				break;
			}

			$this->em->getConnection()->commit();
		} catch (Exception) {
			if ($this->em->getConnection()->isTransactionActive()) {
				$this->em->getConnection()->rollBack();
			}
		}

		$this->cacheService->defaultCache->clean([Cache::Tags => ['features']]);
	}

	public function getOptionsForSelect(): array
	{
		$result = [];

		foreach ($this->getEr()->createQueryBuilder('f')->select('f.id, ft.name')
			         ->innerJoin('f.featureTexts', 'ft', Join::WITH, 'ft.lang = :lang')
			         ->setParameter('lang', $this->translator->getLocale())
			         ->orderBy('f.position')->getQuery()->getArrayResult() as $row) {
			$result[$row['id']] = $row['name'];
		}

		return $result;
	}

	public function sortAlphabetically(): bool
	{
		$data = [];
		foreach ($this->getEr()->createQueryBuilder('f')->select('f.id, ft.name')
			         ->innerJoin('f.featureTexts', 'ft', Join::WITH, 'ft.lang = :lang')
			         ->setParameter('lang', $this->translator->getLocale())
			         ->getQuery()->getArrayResult() as $k => $row) {
			$data[] = [
				'id'   => $row['id'],
				'name' => $row['name'],
			];
		}
		usort($data, static fn($a, $b) => Strings::sortCzech($a['name'], $b['name']));

		$conn = $this->em->getConnection();
		/** @var ClassMetadataInfo $metadata */
		/** @phpstan-ignore-next-line */
		$metadata  = $this->em->getClassMetadata($this->entityClass);
		$tableName = $metadata->getTableName();
		foreach ($data as $k => $v) {
			$conn->update($tableName, [
				'position' => $k,
			], [
				'id' => $v['id'],
			]);
		}

		return true;
	}

	public function getFlatTree(bool $idWithFeature = true): array
	{
		$flat         = [];
		$featureTexts = $this->getOptionsForSelect();
		foreach ($this->featureValues->getOptionsForSelect() as $featureId => $values) {
			$flat[] = [
				'id'     => 'f' . $featureId,
				'name'   => $featureTexts[$featureId],
				'parent' => 0,
			];

			foreach ($values as $valueId => $value) {
				$flat[] = [
					'id'     => ($idWithFeature ? $featureId . '-' : '') . $valueId,
					'name'   => $value,
					'parent' => 'f' . $featureId,
				];
			}
		}

		return $flat;
	}

}
