<?php declare(strict_types = 1);

namespace EshopCatalog\AdminModule\Model;

use Contributte\Translation\Translator;
use Core\Model\Entities\EntityManagerDecorator;
use Doctrine\ORM\Query\Expr\Join;
use EshopCatalog\FrontModule\Model\CacheService;
use EshopCatalog\Model\Entities\DynamicFeatureProduct;
use EshopCatalog\Model\Entities\Feature;
use EshopCatalog\Model\Entities\FeatureProduct;
use EshopCatalog\Model\Entities\FeatureValue;
use EshopCatalog\Model\Entities\FeatureValueTexts;
use EshopCatalog\Model\Entities\Product;
use Exception;
use Nette\Caching\Cache;

class FeaturesTypeService
{
	public function __construct(
		protected EntityManagerDecorator $em,
		protected Translator             $translator,
		protected CacheService           $cacheService,
	)
	{
	}

	public function changeType(string $from, string $to, int $featureId): bool
	{
		$this->em->beginTransaction();
		try {
			$currentProductsValues = [];
			switch ($from) {
				case Feature::TYPE_CHECK:
					$values = [];
					foreach ($this->em->getRepository(FeatureValue::class)->createQueryBuilder('fv')
						         ->select('fv.id, fvt.name')
						         ->innerJoin('fv.featureValueTexts', 'fvt', Join::WITH, 'fvt.lang = :lang')
						         ->where('fv.feature = :feature')
						         ->setParameters([
							         'feature' => $featureId,
							         'lang'    => $this->translator->getLocale(),
						         ])->getQuery()->getArrayResult() as $row) {
						$values[$row['id']] = (string) floatval(str_replace(',', '.', (string) $row['name']));
					}

					if (empty($values)) {
						break;
					}

					foreach ($this->em->getRepository(FeatureProduct::class)->createQueryBuilder('fp')
						         ->select('IDENTITY(fp.product) as product, IDENTITY(fp.featureValue) as value')
						         ->where('fp.featureValue IN (' . implode(',', array_keys($values)) . ')')
						         ->getQuery()->getArrayResult() as $row) {
						$currentProductsValues[$row['product']][$row['value']] = $values[$row['value']];
					}

					$this->em->createQueryBuilder()->delete(FeatureValue::class, 'fv')
						->where('fv.feature = :feature')
						->setParameter('feature', $featureId)
						->getQuery()->execute();
					$this->em->createQueryBuilder()->delete(FeatureProduct::class, 'fp')
						->where('fp.featureValue IN (' . implode(',', array_keys($values)) . ')')
						->getQuery()->execute();

					break;
				case Feature::TYPE_RANGE:
					foreach ($this->em->getRepository(DynamicFeatureProduct::class)->createQueryBuilder('dfp')
						         ->select('IDENTITY(dfp.product) as product, dfp.value')
						         ->where('dfp.feature = :feature')
						         ->setParameter('feature', $featureId)
						         ->getQuery()->getArrayResult() as $row) {
						$currentProductsValues[$row['product']][] = $row['value'];
					}

					$this->em->createQueryBuilder()->delete(DynamicFeatureProduct::class, 'dfp')
						->where('dfp.feature = :feature')
						->setParameter('feature', $featureId)
						->getQuery()->execute();

					break;
			}

			$this->em->flush();

			switch ($to) {
				case Feature::TYPE_CHECK:
					$list  = [];
					$langs = array_map(static fn($v,
					) => explode('_', (string) $v)[0], $this->translator->getAvailableLocales());

					foreach ($currentProductsValues as $product => $values) {
						/** @var Product $product */
						$product = $this->em->getReference(Product::class, $product);

						/** @var Feature $feature */
						$feature = $this->em->getReference(Feature::class, $featureId);

						foreach ($values as $value) {
							if (!isset($list[$value])) {
								$featureValue = new FeatureValue;
								$featureValue->setFeature($feature);
								$featureValue->isPublished = 1;
								$this->em->persist($featureValue);

								foreach ($langs as $l) {
									$featureValueText = new FeatureValueTexts($featureValue, $l);
									$featureValueText->setName($value);
									$this->em->persist($featureValueText);
								}

								$list[$value] = $featureValue;
							}

							$featureProduct = new FeatureProduct($product, $list[$value]);
							$this->em->persist($featureProduct);
						}
					}
					break;
				case Feature::TYPE_RANGE:
					foreach ($currentProductsValues as $product => $values) {
						/** @var Product $product */
						$product = $this->em->getReference(Product::class, $product);

						/** @var Feature $feature */
						$feature = $this->em->getReference(Feature::class, $featureId);

						foreach ($values as $value) {
							$dynamicValue = new DynamicFeatureProduct($product, $feature, (string) $value);
							$this->em->persist($dynamicValue);
						}
					}

					break;
			}

			$this->em->flush();
			$this->em->commit();

			foreach (array_chunk(array_keys($currentProductsValues), 50) as $chunk) {
				$tags = [];
				foreach ($chunk as $productId) {
					$tags[] = 'product/' . $productId;
				}

				$this->cacheService->clean('product', [
					Cache::Tags => $tags,
				]);
			}
		} catch (Exception) {
			if ($this->em->getConnection()->isTransactionActive()) {
				$this->em->rollback();
			}

			return false;
		}

		return true;
	}

}
