<?php declare(strict_types = 1);

namespace EshopCatalog\AdminModule\Model\FormContainers;

use Contributte\Translation\Translator;
use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Helpers\Strings;
use Core\Model\Images\ImageHelper;
use Core\Model\SystemConfig;
use Core\Model\UI\Form\BaseContainer;
use EshopCatalog\FrontModule\Model\CacheService;
use EshopCatalog\Model\Config;
use EshopCatalog\Model\Entities\Product;
use EshopCatalog\Model\Entities\ProductDocument;
use Nette\Http\FileUpload;
use Nette\Utils\ArrayHash;
use Nette\Utils\Validators;

class DocumentContainer
{
	public function __construct(
		protected Translator             $translator,
		protected EntityManagerDecorator $em,
		protected CacheService           $cacheService,
	)
	{
	}

	public function getContainer(bool $multilang = false): BaseContainer
	{
		$container = new BaseContainer();
		$container->addCustomData('template', __DIR__ . '/DocumentContainer.latte');

		for ($i = 0; $i < Config::load('maxDocumentsToUpload', 20); $i++) {
			$documentContainer = new BaseContainer;

			$documentContainer->addText('fileName')
				->setPlaceholder($this->translator->translate('eshopCatalog.documents.fileName'));
			$documentContainer->addUpload('fastFile', 'eshopCatalog.documents.fastUpload');
			$documentContainer->addFilesManager('filePath', 'eshopCatalog.documents.filePath')
				->addSecuredRoot([SystemConfig::load('filesManager.securedRoot')]);
			$documentContainer->addLangsSelect('lang', 'eshopCatalog.documents.lang')
				->setPrompt(null);
			$documentContainer->addHidden('removeLink');
			$documentContainer->addHidden('id');

			$documentContainer->getComponent('lang')
				->setItems(
					[null => 'eshopCatalog.documents.lang'] + $documentContainer->getComponent('lang')
						->getItems(),
				);

			$container->addComponent($documentContainer, 'document_' . $i);
		}

		return $container;
	}

	public function setDefaults(BaseContainer $container, array|ArrayHash $values): void
	{
		$i = 0;
		/** @var BaseContainer $documentContainer */
		foreach ($container->getComponents() as $documentContainer) {
			foreach ($documentContainer->getControls() as $c) {
				if (isset($values[$i][$c->getName()])) {
					$c->setDefaultValue($values[$i][$c->getName()]);
				}
			}
			$i++;
		}
	}

	public function saveData(ArrayHash $data, Product $product): void
	{
		foreach ($data as $document) {
			/** @var FileUpload $file */
			$file = $document['fastFile'];

			if ($file->isOk()) {
				$path = DS . 'images' . DS . Strings::webalize(
						$this->translator->translate('eshopCatalog.documents.baseFolder'),
					) . DS . $file->getSanitizedName();
				$file->move(WWW_DIR . $path);

				if ($file->isImage()) {
					ImageHelper::autoResize(WWW_DIR . $path);
				}

				$document['filePath'] = $path;
			}

			if (!$document['filePath']) {
				continue;
			}

			if (!$document['fileName']) {
				$document['fileName'] = basename((string) $document['filePath']);
			}

			if ($document['lang'] === '') {
				$document['lang'] = null;
			}

			if (Validators::isNone($document['id'])) {
				$this->em->persist(
					new ProductDocument($document['lang'], $document['fileName'], $document['filePath'], $product),
				);
			} else {
				foreach ($product->getDocuments() as $doc) {
					if ($doc->getId() === ((int) $document['id'])) {
						$doc->lang = $document['lang'] ?: null;
						$doc->name = $document['fileName'] ?? null;
						$doc->file = $document['filePath'];
						$this->em->persist($doc);
						break;
					}
				}
			}
		}

		$this->cacheService->removeDocuments([$product->getId()]);
	}

}
