<?php declare(strict_types = 1);

namespace EshopCatalog\AdminModule\Model\FormContainers;

use Contributte\Translation\Translator;
use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\UI\Form\BaseContainer;
use EshopCatalog\AdminModule\Components\Products\ProductForm;
use EshopCatalog\AdminModule\Model\Products;
use EshopCatalog\AdminModule\Model\Repository\RelatedGroups;
use EshopCatalog\Model\Entities\Product;
use EshopCatalog\Model\Entities\RelatedProduct;

class RelatedProductContainer
{
	public function __construct(
		protected Translator             $translator,
		protected EntityManagerDecorator $em,
		protected Products               $products,
		protected RelatedGroups          $relatedGroups,
	)
	{
	}

	public function getContainer(bool $multilang = false): BaseContainer
	{
		$container = new BaseContainer();
		$container->addText('product', 'eshopCatalog.productForm.findProducts')
			->setHtmlAttribute('class', ['productAutocomplete', 'typeahead'])
			->setHtmlAttribute('autocomplete', 'off')
			->setHtmlId('product-finder');
		$container->addCustomData('template', __DIR__ . '/RelatedProductContainer.latte');
		$container->addCustomData('groups', $this->relatedGroups->getOptionsForSelect());

		$container->monitor(ProductForm::class, function(ProductForm $control) use ($container) {
			$product = $control->product;
			$related = [];

			if ($product) {
				foreach ($this->em->createQueryBuilder()
					         ->select('rp, p, pt')
					         ->from(RelatedProduct::class, 'rp')
					         ->innerJoin('rp.product', 'p')
					         ->innerJoin('p.productTexts', 'pt')
					         ->where('rp.origin = :prod')
					         ->setParameter('prod', $product->getId())
					         ->getQuery()->getResult() as $row) {
					/** @var RelatedProduct $row */
					$related[$row->getProduct()->getId()] = [
						'name'  => $row->getProduct()->getText()->name,
						'group' => $row->getGroup()->getId(),
					];
				}
			}

			$container->addCustomData('related', $related);
		});

		return $container;
	}

	/**
	 * @param Product[] $relatedProducts
	 */
	public function setDefaults(BaseContainer $container, array $relatedProducts): void
	{
	}

	public function saveData(array $data, Product $product): void
	{
		$data           = $data['relatedProductsIds'];
		$currentRelated = $product->getRelatedProducts()->toArray();

		unset($data['__id__']);

		foreach (array_diff_key($currentRelated, $data) as $k => $entity) {
			/** @var RelatedProduct $entity */
			$this->em->getConnection()
				->executeStatement('DELETE FROM eshop_catalog__related_product WHERE id = ?', [$entity->getId()]);
			unset($currentRelated[$k]);
		}

		foreach (array_diff_key($data, $currentRelated) as $prodId => $vals) {
			if (!is_numeric($prodId)) {
				continue;
			}

			$group   = $this->relatedGroups->getReference($vals['group']);
			$related = new RelatedProduct($product, $this->products->getReference($prodId), $group);

			$this->em->persist($related);
		}

		foreach ($currentRelated as $k => $entity) {
			$dataRow = $data[$k] ?? null;

			if (!$dataRow) {
				continue;
			}

			if ((int) $dataRow['group'] !== $entity->getGroup()->getId()) {
				$entity->setGroup($this->relatedGroups->getReference((int) $dataRow['group']));
				$this->em->persist($entity);
			}
		}
	}

}
