<?php declare(strict_types = 1);

namespace EshopCatalog\AdminModule\Model\Import;

use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Helpers\BaseService;
use Core\Model\Helpers\Strings;
use Core\Model\Images\ImageHelper;
use Core\Model\Lang\DefaultLang;
use Doctrine\ORM\AbstractQuery;
use Doctrine\ORM\ORMException;
use Doctrine\ORM\Query;
use EshopCatalog\Model\Config;
use EshopCatalog\Model\Entities\Category;
use EshopCatalog\Model\Entities\CategoryProduct;
use EshopCatalog\Model\Entities\CategoryTexts;
use EshopCatalog\Model\Entities\Feature;
use EshopCatalog\Model\Entities\FeatureTexts;
use EshopCatalog\Model\Entities\FeatureValue;
use EshopCatalog\Model\Entities\FeatureValueTexts;
use EshopCatalog\Model\Entities\Manufacturer;
use EshopCatalog\Model\Entities\Product;
use EshopCatalog\Model\Entities\ProductPriceHistory;
use EshopCatalog\Model\Entities\Supplier;
use EshopCatalog\Model\Entities\VatRate;
use Exception;
use Gallery\Model\Entities\Album;
use Gallery\Model\Entities\Image;
use Import\AdminModule\Model\DictionaryResolver;
use Nette\Utils\DateTime;
use Nette\Utils\FileSystem;
use Nette\Utils\Validators;
use Throwable;
use Tracy\Debugger;

class ImportHelper extends BaseService
{
	/**
	 * @var null|array<int, array{
	 *     id: ?int,
	 *     name: ?string,
	 *     entity: ?Manufacturer,
	 *     }>
	 */
	protected ?array $cManufacturers = null;

	/** @var CategoryProduct[] */
	protected array $cCategoryProduct = [];

	/**
	 * @var null|array<int, array{
	 *     id: ?int,
	 *     name: ?string,
	 *     entity: ?Supplier,
	 *     }>
	 */
	protected ?array $cSuppliers = null;

	/** @var Category[] */
	protected array $cCats = [];

	/** @var VatRate[] */
	protected array $cVatRate = [];

	/** @var callable[] */
	public array $onError = [];

	/** @var array */
	public array $featureProducts = [];

	protected array  $cFeatures                    = [];
	protected ?array $cVariants                    = null;
	protected array  $cClearedDynamicFeatureValues = [];

	public function __construct(
		protected EntityManagerDecorator $em,
		protected DictionaryResolver     $dictionaryResolver,
		protected DefaultLang            $defaultLang,
	)
	{
	}

	public function findByEan(string $ean): ?Product
	{
		return $this->em->getRepository(Product::class)->findOneBy(['ean' => $ean, 'isDeleted' => 0]);
	}

	public function findByCodeAndManufacturer(string $code, string $manufacturer): ?Product
	{
		$manufacturer = $this->getManufacturer($manufacturer);

		return $this->em->getRepository(Product::class)->createQueryBuilder('p')
			->where('p.code1 = :code')
			->andWhere('p.isDeleted = 0')
			->andWhere('p.manufacturer = :manu')
			->setParameters([
				'code' => $code,
				'manu' => $manufacturer->getId(),
			])->getQuery()->setMaxResults(1)->getOneOrNullResult();
	}

	public function getManufacturer(string $value): ?Manufacturer
	{
		if ($value === '') {
			return null;
		}

		if ($this->cManufacturers === null) {
			$this->cManufacturers = [];

			foreach ($this->em->getRepository(Manufacturer::class)->createQueryBuilder('m')
				         ->select('m.id, m.name')
				         ->getQuery()->getArrayResult() as $row) {
				$this->cManufacturers[] = [
					'id'     => (int) $row['id'],
					'name'   => (string) $row['name'],
					'entity' => null,
				];
			}
		}

		// Vrátí výrobce pokud je nalezen
		foreach ($this->cManufacturers as $manufacturer) {
			if (strtolower($manufacturer['name']) === strtolower($value)) {
				if (isset($manufacturer['id'])) {
					/** @var Manufacturer $result */
					$result = $this->em->getReference(Manufacturer::class, $manufacturer['id']);

					return $result;
				}
				if (isset($manufacturer['entity'])) {
					return $manufacturer['entity'];
				}
			}
		}

		$manufacturer       = new Manufacturer;
		$manufacturer->name = $value;
		$this->em->persist($manufacturer);
		$this->em->flush($manufacturer);

		$this->cManufacturers[] = [
			'id'     => $manufacturer->getId(),
			'name'   => (string) $value,
			'entity' => $manufacturer,
		];

		return $manufacturer;
	}

	/**
	 * @throws ORMException
	 */
	public function getSupplier(string|int|float $value): ?Supplier
	{
		if ($this->cSuppliers === null) {
			$this->cSuppliers = [];

			foreach ($this->em->getRepository(Supplier::class)->createQueryBuilder('s')
				         ->select('s.id, s.name')
				         ->getQuery()->getArrayResult() as $row) {
				$this->cSuppliers[] = [
					'id'     => (int) $row['id'],
					'name'   => (string) $row['name'],
					'entity' => null,
				];
			}
		}

		// Vrátí dodavatele pokud je nalezen
		foreach ($this->cSuppliers as $supplier) {
			if (strtolower((string) $supplier['name']) == strtolower((string) $value)) {
				if ($supplier['id']) {
					/** @var Supplier $result */
					$result = $this->em->getReference(Supplier::class, $supplier['id']);

					return $result;
				}
				if (isset($supplier['entity'])) {
					return $supplier['entity'];
				}
			}
		}

		$supplier = new Supplier((string) $value);
		$this->em->persist($supplier)->flush($supplier);
		$this->cSuppliers[] = [
			'id'     => $supplier->getId(),
			'name'   => (string) $value,
			'entity' => $supplier,
		];

		return $supplier;
	}

	public function findCategory(int $id): ?Category
	{
		/** @var ?Category $result */
		$result = $this->em->getRepository(Category::class)->find($id);

		return $result;
	}

	/**
	 * @param int|string|string[] $deep
	 *
	 * @throws Throwable
	 */
	public function getDeepCategory(
		Category         $baseCat,
		int|string|array $deep,
		string           $lang,
		int              $isPublished = 0,
	): ?Category
	{
		if (!is_array($deep)) {
			$deep = [$deep];
		}

		$parent = $baseCat;
		$cat    = null;

		$this->cCats = [];
		$deepCount   = count($deep);
		for ($i = 0; $i < $deepCount; $i++) {
			$cat      = null;
			$name     = trim((string) $deep[$i]);
			$cCatsKey = $i . '_' . $name;

			if (isset($this->cCats[$cCatsKey])) {
				return $this->cCats[$cCatsKey];
			}

			$found = $this->em->getRepository(CategoryTexts::class)->createQueryBuilder('ct')
				->select('IDENTITY(ct.id) as id')
				->where('ct.name = :name')->andWhere('ct.lang = :lang')
				->innerJoin('ct.id', 'c', Query\Expr\Join::WITH, 'c.parent = :parent')
				->setParameters([
					'name'   => $name,
					'lang'   => $lang,
					'parent' => $parent->getId(),
				])
				->setMaxResults(1)->getQuery()->getOneOrNullResult(AbstractQuery::HYDRATE_ARRAY);

			if ($found) {
				/** @var Category $cat */
				$cat = $this->em->getRepository(Category::class)->find($found['id']);

				$this->cCats[$cCatsKey] = $cat;
			} else {
				$cat              = new Category;
				$cat->isPublished = $isPublished;
				$cat->setParent($parent);
				$this->em->persist($cat)->flush($cat);

				$ct = new CategoryTexts($cat, $lang);
				$ct->setName($name);
				$cat->setCategoryText($ct);
				$this->em->persist($ct)->flush($ct);
				$this->cCats[$cCatsKey] = $cat;
			}
			$parent = $cat;
		}

		return $cat;
	}

	public function checkCategoryProductExist(int|string $productId, int|string $categoryId): bool
	{
		$key = "$productId-$categoryId";
		$cat = $this->cCategoryProduct[$key] ?? null;

		if (!$cat) {
			$cat = $this->em->getRepository(CategoryProduct::class)->findOneBy([
				'product'  => $productId,
				'category' => $categoryId,
			]);

			if ($cat) {
				$this->cCategoryProduct[$key] = $cat;
			}
		}

		return $cat ? true : false;
	}

	public function getCategoryProduct(Product $product, Category $category): ?CategoryProduct
	{
		$key = $product->getId() . '-' . $category->getId();
		$cat = $this->cCategoryProduct[$key] ?? null;

		if (!$cat) {
			$cat = $this->em->getConnection()
				->fetchOne(
					"SELECT id_product FROM eshop_catalog__category_product WHERE id_product = ? AND id_category = ?",
					[$product->getId(), $category->getId()],
				);

			if (!$cat) {
				$this->em->getConnection()->insert('eshop_catalog__category_product', [
					'id_product'  => $product->getId(),
					'id_category' => $category->getId(),
				]);
			}

			$cat = $this->em->getRepository(CategoryProduct::class)->findOneBy([
				'product'  => $product,
				'category' => $category,
			]);

			$this->cCategoryProduct[$key] = $cat;
		}

		return $cat;
	}

	public function getGallery(Product $product): Album
	{
		if (!$product->getGallery()) {
			$gallery = new Album(UPLOADS_PATH . '/products');
			$this->em->persist($gallery);
			$this->em->flush($gallery);

			return $gallery;
		}

		return $product->getGallery();
	}

	public function getImages(array|string $data, ?string $urlPrefix = ''): array
	{
		$imgs = [];
		$loop = static function($rows) use (&$loop, &$imgs, $urlPrefix) {
			if (is_string($rows)) {
				$imgs[] = $urlPrefix . $rows;
			} else {
				if (isset($rows['file'], $rows['fullFile'])) {
					unset($rows['file']);
				}

				foreach ($rows as $v) {
					if (is_array($v)) {
						$loop($v);
					}

					if (is_string($v) && $v != '') {
						$imgs[] = $urlPrefix . $v;
					}
				}
			}
		};

		$loop($data);

		return $imgs;
	}

	public function addImage(Album $gallery, string $url): bool
	{
		try {
			// Kontrola existence souboru
			$curl = curl_init($url);
			if (!$curl) {
				return false;
			}

			curl_setopt($curl, CURLOPT_NOBODY, true);
			$result = curl_exec($curl);
			$ret    = false;

			if ($result !== false) {
				$statusCode = curl_getinfo($curl, CURLINFO_HTTP_CODE);
				if (in_array($statusCode, [200, 301, 302])) {
					$ret = true;
				}
			}
			curl_close($curl);

			if (!$ret) {
				return false;
			}

			$images    = $gallery->getImages();
			$basename  = Strings::webalize(pathinfo($url, PATHINFO_FILENAME));
			$extension = pathinfo($basename, PATHINFO_EXTENSION);
			$exist     = false;

			if (strtolower($extension) === 'tiff') {
				return false;
			}

			if (!in_array(strtolower($extension), ['jpg', 'jpeg', 'png'])) {
				$basename = pathinfo($basename, PATHINFO_FILENAME) . '.jpg';
			}

			/** @var Image $img */
			foreach ($images->toArray() as $img) {
				if (!file_exists($img->getFile())) {
					$this->em->getConnection()->delete('gallery__image', [
						'id' => $img->getId(),
					]);
					continue;
				}

				$fn = explode('_', $img->getFilename(), 2);

				if (isset($fn[1])) {
					$fn[1] = str_replace('-jpg', '', $fn[1]);
					if (basename($fn[1]) == $basename) {
						$exist = true;
						break;
					}
				}
			}

			if (!$exist) {
				// TODO dát do služby a nastavit i v imagesZone komponentě
				/** @var int|false $lastPosition */
				$lastPosition = $this->em->getConnection()
					->fetchOne(
						"SELECT position FROM gallery__image WHERE album_id = :album",
						[':album' => $gallery->getId()],
					);
				if (!$lastPosition && $lastPosition != 0) {
					$position = 0;
				} else {
					$position = $lastPosition + 1;
				}
				$this->em->getConnection()->insert('gallery__image', [
					'filename'     => $basename,
					'path'         => $gallery->generatePath(),
					'album_id'     => $gallery->getId(),
					'position'     => $position,
					'is_published' => 1,
					'is_cover'     => 0,
				]);
				$imageId = $this->em->getConnection()->lastInsertId();

				$filename = $imageId . '_' . $basename;
				$this->em->getConnection()->update('gallery__image', [
					'filename' => $filename,
				], ['id' => $imageId]);

				$dir = WWW_DIR . $gallery->generatePath();
				FileSystem::createDir($dir);
				copy($url, $dir . '/' . $filename);
				ImageHelper::autoResize($dir . '/' . $filename);

				$this->onError[] = function() use ($dir, $filename) {
					FileSystem::delete($dir . '/' . $filename);

					if (count((array) scandir($dir)) === 2) {
						FileSystem::delete($dir);

						$galleryId = (int) basename($dir);
						$gallery   = $this->em->getRepository(Album::class)->find($galleryId);
						if ($gallery) {
							$this->em->remove($gallery);
						}
					}
				};
			}
		} catch (Exception $e) {
			Debugger::log($e, 'import');
		}

		return true;
	}

	/**
	 * @param Album[] $galleries
	 */
	public function moveGalleryImage(array $galleries): void
	{
		// @Roman - smazat tuto funkci?
		return;
		//		foreach ($galleries as $gallery) {
		//			if (!$gallery || $gallery instanceof Album === false) {
		//				continue;
		//			}
		//
		//			$gallery = $this->em->getRepository(Album::class)->find($gallery->getId());
		//
		//			if (!$gallery) {
		//				continue;
		//			}
		//
		//			$path = $gallery->generatePath();
		//			$dir  = WWW_DIR . $path;
		//			FileSystem::createDir($dir);
		//
		//			foreach ($gallery->getImages() as $img) {
		//				$tmp = explode('_', $img->getFilename());
		//
		//				if ($tmp[0] != '') {
		//					continue;
		//				}
		//
		//				$oldFile       = $img->getFile();
		//				$img->path     = $gallery->generatePath();
		//				$img->filename = $img->getId() . $img->getFilename();
		//
		//				rename($oldFile, $img->getFile());
		//				$this->em->persist($img);
		//			}
		//		}
		//
		//		$this->em->flush();
	}

	public function categoriesList(): array
	{
		$list = [];

		$qb = $this->em->getRepository(Category::class)->createQueryBuilder('c')
			->select('c.id', 'c.lvl', 'ct.name')
			->join('c.categoryTexts', 'ct', 'WITH', 'ct.lang = :lang')
			->setParameter('lang', $this->defaultLang->locale)
			->where('c.lvl > 0')
			->orderBy('c.root')->addOrderBy('c.lft');

		foreach ($qb->getQuery()->getArrayResult() as $row) {
			$deep = '';

			for ($i = 1; $i < $row['lvl']; $i++) {
				$deep = '---' . $deep;
			}

			$list[$row['id']] = $deep . ' ' . $row['name'];
		}

		return $list;
	}

	public function vatRateList(): array
	{
		if (!$this->cVatRate) {
			$qb = $this->em->getRepository(VatRate::class)->createQueryBuilder('vr');

			foreach ($qb->getQuery()->getArrayResult() as $row) {
				$keys = [
					$row['rate'],
					$row['rate'] . '.00',
					($row['rate'] / 100),
					($row['rate'] + 100) / 100];

				foreach ($keys as $k) {
					$this->cVatRate[str_replace(',', '.', (string) $k)] = $row['id'];
				}
			}
		}

		return $this->cVatRate;
	}

	/**
	 * @throws ORMException
	 */
	public function getVatRate(int|string $val): ?VatRate
	{
		$list = $this->vatRateList();

		/** @var VatRate|null $result */
		$result = isset($list[$val]) ? $this->em->getReference(VatRate::class, $list[$val]) : null;

		return $result;
	}

	public function getFeatures(string $lang): array
	{
		if (!$this->cFeatures) {
			$featuresIdName = [];
			foreach ($this->em->getRepository(Feature::class)->createQueryBuilder('f')
				         ->select('f.id, ft.name, f.type, f.unit')
				         ->join('f.featureTexts', 'ft', 'WITH', 'ft.lang = :lang')
				         ->setParameter('lang', $lang)->getQuery()->getArrayResult() as $v) {
				$name                     = mb_strtolower((string) $v['name']);
				$this->cFeatures[$name]   = [
					'id'     => $v['id'],
					'values' => [],
					'type'   => $v['type'],
					'unit'   => $v['unit'],
				];
				$featuresIdName[$v['id']] = $name;
			}

			foreach ($this->em->getRepository(FeatureValue::class)->createQueryBuilder('fv')
				         ->select('fv.id, fvt.name, IDENTITY(fv.feature) as featureId')
				         ->join('fv.featureValueTexts', 'fvt', 'WITH', 'fvt.lang = :lang')
				         ->setParameter('lang', $lang)->getQuery()->getArrayResult() as $v) {
				$name                                                               = mb_strtolower(
					(string) $v['name'],
				);
				$this->cFeatures[$featuresIdName[$v['featureId']]]['values'][$name] = $v['id'];
			}
			unset($featuresIdName);
		}

		return $this->cFeatures;
	}

	public function parametersProccess(Product $product, array $params, string $lang): void
	{
		$features = $this->getFeatures($lang);

		foreach ($params as $key => $value) {
			if (!$value || !$key || is_numeric($key)) {
				continue;
			}

			if (is_array($value)) {
				$this->parametersProccess($product, $value, $lang);
				continue;
			}

			if (mb_strtoupper($key) == $key) {
				$key = ucfirst(mb_strtolower($key));
			}

			$key   = $this->dictionaryResolver->findValue($key, $lang);
			$value = $this->dictionaryResolver->findValue($value, $lang);

			$lKey   = mb_strtolower($key);
			$lValue = mb_strtolower($value);

			if (Validators::isNone($lValue) || Validators::isNone($lKey)) {
				continue;
			}

			if ($features[$lKey]['type'] === Feature::TYPE_RANGE) {
				if ($features[$lKey]['unit']) {
					$value = trim((string) substr((string) $value, 0, -strlen((string) $features[$lKey]['unit'])));
				}

				$this->addDynamicFeatureValueToProduct($product, (int) $features[$lKey]['id'], $value);
			} else {

				if (!isset($features[$lKey])) {
					$feature                      = $this->createFeature($key, $lang);
					$this->cFeatures[$lKey]['id'] = $feature->getId();
					$features[$lKey]['id']        = $feature->getId();
				}

				if (!isset($features[$lKey]['values'][$lValue])) {
					$featureValue                              = $this->createFeatureValue(
						$features[$lKey]['id'],
						$lValue,
						$lang,
					);
					$this->cFeatures[$lKey]['values'][$lValue] = $featureValue->getId();
					$features[$lKey]['values'][$lValue]        = $featureValue->getId();
				}

				$this->addFeatureValueToProduct($product, $features[$lKey]['id'], $features[$lKey]['values'][$lValue]);
			}
		}
	}

	public function createFeature(string|int $value, string $lang): Feature
	{
		$feature = new Feature;
		$feature->setIsPublished(1);
		$feature->setUseAsFilter(0);
		$this->em->persist($feature);

		$featureText       = new FeatureTexts($feature, $lang);
		$featureText->name = (string) $value;
		$this->em->persist($featureText);

		$this->em->flush([$feature, $featureText]);

		return $feature;
	}

	/**
	 *
	 * @throws ORMException
	 */
	public function createFeatureValue(
		int|Feature $featureId,
		string|int  $value,
		string      $lang,
	): FeatureValue
	{
		if (is_numeric($featureId)) {
			/** @var Feature $feature */
			$feature = $this->em->getReference(Feature::class, $featureId);
		} else {
			$feature = $featureId;
		}

		$featureValue = new FeatureValue;
		$featureValue->setFeature($feature);
		$this->em->persist($featureValue);

		$featureValueText       = new FeatureValueTexts($featureValue, $lang);
		$featureValueText->name = (string) $value;
		$this->em->persist($featureValueText);

		$this->em->flush([$featureValue, $featureValueText]);

		return $featureValue;
	}

	/**
	 *
	 * @throws \Doctrine\DBAL\Exception
	 */
	public function addFeatureValueToProduct(Product $product, int|string $feature, int|string $featureValue): void
	{
		foreach ($product->getFeatureProducts() as $fp) {
			if ($fp->getIdFeatureValue() == $featureValue) {
				return;
			}
		}

		$this->em->getConnection()
			->executeStatement(
				"INSERT IGNORE INTO eshop_catalog__feature_product (id_product, id_feature_value) VALUES (" . $product->getId(
				) . ", " . $featureValue . ")",
			);
	}

	/**
	 * @throws \Doctrine\DBAL\Exception
	 */
	public function addDynamicFeatureValueToProduct(Product $product, int $feature, int|string $value): void
	{
		if (!isset($this->cClearedDynamicFeatureValues[$product->getId()])) {
			$count = $this->em->getConnection()
				->executeStatement(
					"DELETE FROM eshop_catalog__dynamic_feature_product WHERE id_product = " . $product->getId(),
				);

			$this->cClearedDynamicFeatureValues[$product->getId()] = (int) $count;
		}

		$this->em->getConnection()->insert('eshop_catalog__dynamic_feature_product', [
			'id_product' => $product->getId(),
			'id_feature' => $feature,
			'value'      => (string) $value,
		]);
	}

	public function setProductPrice(Product $product, float $value, string $column = 'price'): void
	{
		if ($product->getMoreDataValue('stopImportPrice') == true) {
			return;
		}

		if (Config::load('enablePriceHistory') && $column === 'price' && (float) $product->price !== $value) {
			$priceHistory = new ProductPriceHistory(
				$product,
				(float) $product->price,
				$value,
				'import',
			);
			$this->em->persist($priceHistory);
		}

		$product->$column = Strings::formatEntityDecimal($value);
	}

	public function checkValueIs(string $valueIs, int|string|float $val, int|string|float $compare): bool
	{
		if ($valueIs !== '=') {
			$val     = $this->extractInt($val);
			$compare = $this->extractInt($compare);
		}

		return match ($valueIs) {
			'<' => $val < $compare,
			'<=' => $val <= $compare,
			'=' => $val == $compare,
			'>=' => $val >= $compare,
			'>' => $val > $compare,
			default => false,
		};
	}

	public function extractInt(int|string|float $val): int
	{
		if (!$val) {
			return 0;
		}

		if (is_numeric($val)) {
			return (int) $val;
		}

		preg_match('/\d+/', $val, $matches);
		if (isset($matches[0]) && is_numeric($matches[0])) {
			return (int) $matches[0];
		}

		return 0;
	}

	public function validateEAN(string $ean): string
	{
		return substr($ean, -13);
	}

	protected function loadVariants(): array
	{
		if ($this->cVariants === null) {
			$this->cVariants = [];

			foreach ($this->em->getConnection()
				         ->fetchAllAssociative(
					         "SELECT product_id, variant_id FROM eshop_catalog__product_variant",
				         ) as $row) {
				$this->cVariants[$row['variant_id']][$row['product_id']] = $row['product_id'];
			}
		}

		return $this->cVariants;
	}

	/**
	 * @throws \Doctrine\DBAL\Exception
	 */
	public function addToVariant(int|string $variantId, int $productId): void
	{
		$exists   = $this->loadVariants();
		$baseData = [
			'product_id'      => $productId,
			'variant_id'      => $variantId,
			'created_default' => (new DateTime)->format('Y-m-d H:i:s'),
			'use_name'        => 1,
		];

		if (isset($exists[$variantId][$productId])) {
			return;
		}

		if (isset($exists[$variantId])) {
			$this->em->getConnection()->insert('eshop_catalog__product_variant', $baseData);
		} else {
			$baseData['is_default'] = 1;
			$this->em->getConnection()->insert('eshop_catalog__product_variant', $baseData);
		}

		$exists[$variantId][$productId]          = $productId;
		$this->cVariants[$variantId][$productId] = $productId;
	}

}
