<?php declare(strict_types = 1);

namespace EshopCatalog\AdminModule\Presenters;

use Core\Model\UI\Form\BaseForm;
use EshopCatalog\AdminModule\Components\Manufacturers\IManufacturerFormFactory;
use EshopCatalog\AdminModule\Components\Manufacturers\IManufacturersGridFactory;
use EshopCatalog\AdminModule\Components\Manufacturers\ManufacturerForm;
use EshopCatalog\AdminModule\Components\Manufacturers\ManufacturersGrid;
use EshopCatalog\AdminModule\Components\Suppliers\ISupplierFormFactory;
use EshopCatalog\AdminModule\Components\Suppliers\ISuppliersGridFactory;
use EshopCatalog\AdminModule\Components\Suppliers\SupplierForm;
use EshopCatalog\AdminModule\Components\Suppliers\SuppliersGrid;
use EshopCatalog\Model\Entities\Manufacturer;
use EshopCatalog\Model\Entities\Supplier;
use Nette\Application\AbortException;
use Nette\Application\BadRequestException;

class ManufacturersSuppliersPresenter extends BasePresenter
{

	/*******************************************************************************************************************
	 * ==================================   Actions
	 */

	public function actionDefault(): void
	{
		$this->setTitle($this->translator->translate('eshopCatalog.title.manufacturersSuppliers'));
		$this->setHeader('eshopCatalog.title.manufacturersSuppliers', 'fa fa-list');

		$this['navigation']->setData(['header' => [
			[
				'title' => 'eshopCatalog.menu.addManufacturer',
				'link'  => 'ManufacturersSuppliers:addManufacturer',
				'ico'   => 'plus',
			], [
				'title' => 'eshopCatalog.menu.addSupplier',
				'link'  => 'ManufacturersSuppliers:addSupplier',
				'ico'   => 'plus',
			],
		]]);
	}

	public function actionAddManufacturer(): void
	{
		$this->setTitle($this->translator->translate('eshopCatalog.title.newManufacturer'));
		$this->setHeader('eshopCatalog.title.newManufacturer', 'fa fa-list-alt');
	}

	public function actionAddSupplier(): void
	{
		$this->setTitle($this->translator->translate('eshopCatalog.title.newSupplier'));
		$this->setHeader('eshopCatalog.title.newSupplier', 'fa fa-list-alt');
	}

	/**
	 * @throws BadRequestException
	 */
	public function actionEditManufacturer(int|string $id): void
	{
		$manufacturer = $this->em->getRepository(Manufacturer::class)->find($id);

		if (!$manufacturer) {
			$this->error();
		}

		$this->setTitle($this->translator->translate('eshopCatalog.title.editManufacturer'));
		$this->setHeader('eshopCatalog.title.editManufacturer', 'fa fa-list-alt');
	}

	/**
	 * @throws BadRequestException
	 */
	public function actionEditSupplier(int|string $id): void
	{
		$supplier = $this->em->getRepository(Supplier::class)->find($id);

		if (!$supplier) {
			$this->error();
		}

		$this->setTitle($this->translator->translate('eshopCatalog.title.editSupplier'));
		$this->setHeader('eshopCatalog.title.editSupplier', 'fa fa-list-alt');
	}

	/**
	 * @throws AbortException
	 */
	public function actionClose(int|string $id): never
	{
		$this->redirect('Default:');
	}

	/*******************************************************************************************************************
	 * ==================================   Components
	 */

	protected function createComponentManufacturerForm(IManufacturerFormFactory $manufacturer): ManufacturerForm
	{
		$control = $manufacturer->create();
		if ($this->getParameter('id')) {
			$control->setManufacturer($this->getParameter('id'));
		}

		$control['form']->onSuccessSave[]         = function(BaseForm $form): never {
			$this->redirect('ManufacturersSuppliers:editManufacturer', $form->getCustomData('manufacturerId'));
		};
		$control['form']->onSuccessSaveAndClose[] = function(): never {
			$this->redirect('ManufacturersSuppliers:default');
		};
		$control['form']->onCancel[]              = function(): never {
			$this->redirect('ManufacturersSuppliers:default');
		};

		return $control;
	}

	protected function createComponentManufacturersGrid(IManufacturersGridFactory $factory): ManufacturersGrid
	{
		return $factory->create();
	}

	protected function createComponentSupplierForm(ISupplierFormFactory $suppliers): SupplierForm
	{
		$control = $suppliers->create();
		if ($this->getParameter('id')) {
			$control->setSupplier($this->getParameter('id'));
		}

		$control['form']->onSuccessSave[]         = function(BaseForm $form): never {
			$this->redirect('ManufacturersSuppliers:editSupplier', $form->getCustomData('supplierId'));
		};
		$control['form']->onSuccessSaveAndClose[] = function(): never {
			$this->redirect('ManufacturersSuppliers:default');
		};
		$control['form']->onCancel[]              = function(): never {
			$this->redirect('ManufacturersSuppliers:default');
		};

		return $control;
	}

	protected function createComponentSuppliersGrid(ISuppliersGridFactory $factory): SuppliersGrid
	{
		return $factory->create();
	}

}
