<?php declare(strict_types = 1);

namespace EshopCatalog\AdminModule\Presenters;

use Core\Model\UI\Form\BaseForm;
use EshopCatalog\AdminModule\Components\Products\IProductFormFactory;
use EshopCatalog\AdminModule\Components\Products\IProductsCategoryFormFactory;
use EshopCatalog\AdminModule\Components\Products\IProductsFeatureFormFactory;
use EshopCatalog\AdminModule\Components\Products\IProductsGridFactory;
use EshopCatalog\AdminModule\Components\Products\IProductsSpeditionsFormFactory;
use EshopCatalog\AdminModule\Components\Products\IProductTagsFormFactory;
use EshopCatalog\AdminModule\Components\Products\IRelatedProductsFormFactory;
use EshopCatalog\AdminModule\Components\Products\ISetAsVariantForFactory;
use EshopCatalog\AdminModule\Components\Products\ProductForm;
use EshopCatalog\AdminModule\Components\Products\ProductsCategoryForm;
use EshopCatalog\AdminModule\Components\Products\ProductsFeatureForm;
use EshopCatalog\AdminModule\Components\Products\ProductsGrid;
use EshopCatalog\AdminModule\Components\Products\ProductsSpeditionsForm;
use EshopCatalog\AdminModule\Components\Products\ProductTagsForm;
use EshopCatalog\AdminModule\Components\Products\RelatedProductsForm;
use EshopCatalog\AdminModule\Components\Products\SetAsVariantsFor;
use EshopCatalog\Model\Entities\Product;
use Nette\Application\BadRequestException;

class ProductsPresenter extends BasePresenter
{

	/*******************************************************************************************************************
	 * ==================================   Actions
	 */

	public function actionDefault(): void
	{
		$this->setTitle($this->translator->translate('eshopCatalog.title.products'));
		$this->setHeader('eshopCatalog.title.products', 'fa fa-list');

		$this->getSession('productForm')->remove();

		$this['navigation']->setData(['header' => [
			[
				'title' => 'eshopCatalog.menu.addProduct',
				'link'  => 'Products:add',
				'ico'   => 'plus',
			],
		]]);
	}

	public function actionAdd(): void
	{
		$this->setHeader('eshopCatalog.title.newProduct', 'fa fa-list-alt');
	}

	/**
	 * @throws BadRequestException
	 */
	public function actionEdit(int|string $id): void
	{
		/** @var Product|null $product */
		$product = $this->em->getRepository(Product::class)->findOneBy(['id' => (int) $id, 'isDeleted' => 0]);

		if (!$product) {
			throw new BadRequestException;
		}

		$title = 'editProduct';
		if ($product->isVariant()) {
			if ($product->isVariant()->isDefault) {
				$title = 'editDefaultProductVariant';
			} else {
				$title = 'editProductVariant';
			}
		}

		$this->setHeader(
			$this->t('eshopCatalog.title.' . $title) . ': ' . $product->getText($this->translator->getLocale())->name,
			'fa fa-list-alt',
			'(ID: ' . $product->getId() . ')',
		);
	}

	public function actionEditCategories(string $ids): void
	{
		$this->setHeader('eshopCatalog.title.editProductsCategories', 'fa fa-list-alt');
	}

	public function actionEditSpeditions(string $ids): void
	{
		$this->setHeader('eshopCatalog.title.editSpeditions', 'fa fa-list-alt');
	}

	public function actionEditFeatures(string $ids): void
	{
		$this->setHeader('eshopCatalog.title.editProductsFeatures', 'fa fa-list-alt');
	}

	public function actionClose(int $id): never
	{
		$this->redirect('Default:');
	}

	public function actionAssignRelatedProducts(string $ids): void
	{
		$this->setHeader('eshopCatalog.title.assignRelatedProducts', 'fa fa-project-diagram');
	}

	public function actionSetTags(string $ids): void
	{
		$this->setHeader('eshopCatalog.title.setTags', 'fa fa-list-alt');
	}

	public function actionSetAsVariantsFor(string $ids): void
	{
		$this->setHeader('eshopCatalog.title.setAsVariantsFor');
	}

	/*******************************************************************************************************************
	 * ==================================   Components
	 */

	protected function createComponentProductForm(IProductFormFactory $product): ProductForm
	{
		$control = $product->create();
		if ($this->getParameter('id')) {
			$control->setProduct($this->getParameter('id'));
		}

		$control['form']->onSuccessSave[]         = function(BaseForm $form): never {
			$this->redirect('Products:edit', $form->getCustomData('productId'));
		};
		$control['form']->onSuccessSaveAndClose[] = function(BaseForm $form): never {
			$this->redirect('Products:default');
		};
		$control['form']->onCancel[]              = function(BaseForm $form): never {
			$this->redirect('Products:default');
		};

		return $control;
	}

	protected function createComponentProductsGrid(IProductsGridFactory $factory): ProductsGrid
	{
		return $factory->create();
	}

	protected function createComponentProductsCategoryForm(IProductsCategoryFormFactory $factory): ProductsCategoryForm
	{
		$control = $factory->create();
		if ($this->getParameter('ids')) {
			$control->setProducts($this->getParameter('ids'));
		}

		$control['form']->onSuccessSave[]         = function(BaseForm $form): never {
			$this->redirect('Products:editCategories', $form->getCustomData('productIds'));
		};
		$control['form']->onSuccessSaveAndClose[] = function(BaseForm $form): never {
			$this->redirect('Products:default');
		};
		$control['form']->onCancel[]              = function(BaseForm $form): never {
			$this->redirect('Products:default');
		};

		return $control;
	}

	protected function createComponentProductsSpeditionsForm(IProductsSpeditionsFormFactory $factory,
	): ProductsSpeditionsForm
	{
		$control = $factory->create();
		if ($this->getParameter('ids')) {
			$control->setProducts($this->getParameter('ids'));
		}

		$control['form']->onSuccessSave[]         = function(BaseForm $form): never {
			$this->redirect('Products:editSpeditions', $form->getCustomData('productIds'));
		};
		$control['form']->onSuccessSaveAndClose[] = function(BaseForm $form): never {
			$this->redirect('Products:default');
		};
		$control['form']->onCancel[]              = function(BaseForm $form): never {
			$this->redirect('Products:default');
		};

		return $control;
	}

	protected function createComponentProductsFeatureForm(IProductsFeatureFormFactory $factory): ProductsFeatureForm
	{
		$control = $factory->create();
		if ($this->getParameter('ids')) {
			$control->setProducts($this->getParameter('ids'));
		}

		$control['form']->onSuccessSave[]         = function(BaseForm $form): never {
			$this->redirect('Products:editFeatures', $form->getCustomData('productIds'));
		};
		$control['form']->onSuccessSaveAndClose[] = function(BaseForm $form): never {
			$this->redirect('Products:default');
		};
		$control['form']->onCancel[]              = function(BaseForm $form): never {
			$this->redirect('Products:default');
		};

		return $control;
	}

	protected function createComponentRelatedProductsForm(IRelatedProductsFormFactory $factory): RelatedProductsForm
	{
		$control = $factory->create();

		$control['form']->onSuccess[] = function(BaseForm $form): never {
			$this->redirect('Products:default');
		};

		return $control;
	}

	protected function createComponentProductTagsForm(IProductTagsFormFactory $factory): ProductTagsForm
	{
		$control = $factory->create();

		$control['form']->onSuccessSave[]         = function(BaseForm $form): never {
			$this->redirect('Products:setTags', $form->getCustomData('productIds'));
		};
		$control['form']->onSuccessSaveAndClose[] = function(BaseForm $form): never {
			$this->redirect('Products:default');
		};
		$control['form']->onCancel[]              = function(BaseForm $form): never {
			$this->redirect('Products:default');
		};

		return $control;
	}

	protected function createComponentSetAsVariantForForm(ISetAsVariantForFactory $factory): SetAsVariantsFor
	{
		$control = $factory->create();

		$control['form']->onSuccessSave[]         = function(BaseForm $form): never {
			$this->redirect('Products:setAsVariantsFor', $form->getCustomData('productIds'));
		};
		$control['form']->onSuccessSaveAndClose[] = function(BaseForm $form): never {
			$this->redirect('Products:default');
		};
		$control['form']->onCancel[]              = function(BaseForm $form): never {
			$this->redirect('Products:default');
		};

		return $control;
	}

}
