<?php declare(strict_types = 1);

namespace EshopCatalog\ApiModule\Api\V1\Controllers;

use Apitte\Core\Annotation\Controller\Method;
use Apitte\Core\Annotation\Controller\Path;
use Apitte\Core\Annotation\Controller\RequestParameter;
use Apitte\Core\Annotation\Controller\RequestParameters;
use Apitte\Core\Http\ApiRequest;
use Apitte\Core\Http\ApiResponse;
use Contributte\Translation\Translator;
use Core\Model\Entities\QueryBuilder;
use Core\Model\Parameters;
use Doctrine\ORM\AbstractQuery;
use Doctrine\ORM\Query;
use EshopCatalog\AdminModule\Model\Products as AdminProducts;
use EshopCatalog\FrontModule\Model\Dao\Product;
use EshopCatalog\FrontModule\Model\Products;
use EshopCatalog\FrontModule\Model\ProductsFacade;
use EshopCatalog\Model\Config;
use Nette\Application\LinkGenerator;
use Nette\Utils\Arrays;
use Nette\Utils\Validators;
use Throwable;

/**
 * @Path("/product")
 */
class ProductsController extends BaseController
{
	public function __construct(
		protected Products       $productsDaoService,
		protected ProductsFacade $productsFacade,
		protected AdminProducts  $products,
		protected Translator     $translator,
		protected LinkGenerator  $linkGenerator,
	)
	{
		ProductsFacade::setMode(ProductsFacade::MODE_CHECKOUT);
	}

	/**
	 * @Path("/")
	 * @Method("GET")
	 * @RequestParameters({
	 *      @RequestParameter(name="start", type="int", description="start", required=false, in="query"),
	 *      @RequestParameter(name="limit", type="int", description="limit", required=false, in="query"),
	 *      @RequestParameter(name="filter", type="string", description="filter", required=false, in="query")
	 * })
	 */
	public function index(ApiRequest $request, ApiResponse $response): array
	{
		$offset = $request->getQueryParam('offset', 0);
		$limit  = $request->getQueryParam('limit', 0);
		$sort   = $request->getQueryParam('sort', 'ASC');

		$productIds = $this->productsDaoService->getProductsIdAll($offset, $limit, $sort);

		return $productIds ? array_values($this->productsFacade->getProducts($productIds)) : [];
	}

	/**
	 * @return Product[]
	 * @throws Throwable
	 * @Path("/get/{id}")
	 * @Method("GET")
	 * @RequestParameters({
	 *      @RequestParameter(name="id", type="int")
	 * })
	 */
	public function product(ApiRequest $request, ApiResponse $response): array
	{
		$id      = $request->getParameter('id');
		$product = $this->productsFacade->getProduct($id);

		if (!$product) {
			$this->sendError($response, "Product with this id doesn´t exist");
		}

		return [$product];
	}

	/**
	 * @Path("/searchByCode/{code}")
	 * @Method("GET")
	 * @RequestParameters({
	 *      @RequestParameter(name="code", type="string", required=true),
	 *      @RequestParameter(name="userId", type="int", description="userId", required=false, in="query")
	 * })
	 */
	public function searchByCode(ApiRequest $request, ApiResponse $response): ApiResponse
	{
		$query  = urldecode((string) $request->getParameter('code'));
		$userId = $request->getQueryParam('userId', null);
		$qb     = $this->prepareSearchQueryBuilder();
		$qb->andWhere($qb->expr()->eq('p.ean', $query));

		$result = $qb->getQuery()->setMaxResults(1)->getOneOrNullResult(AbstractQuery::HYDRATE_ARRAY);

		if (!$result) {
			return $response->writeJsonBody(['status' => 'success', 'data' => null]);
		}

		$result['link'] = '#';

		if (!Validators::isNone($result['coverPath']) && !Validators::isNone($result['coverFilename'])) {
			$result['path']     = $result['coverPath'];
			$result['filename'] = $result['coverFilename'];
		}
		unset($result['coverPath'], $result['coverFilename']);

		$rows = [$result];
		if (class_exists('EshopCheckout\DI\EshopCheckoutExtension') && Parameters::load(
				'eshopCheckout.allowPriceLevels',
				false,
			) && !is_null($userId)) {
			$this->loadPriceLevels($rows, (int) $userId);
		}

		return $response->writeJsonBody(Arrays::first((array) $rows));
	}

	/**
	 * @Path("/search/{query}")
	 * @Method("GET")
	 * @RequestParameters({
	 *      @RequestParameter(name="query", type="string", required=true),
	 *      @RequestParameter(name="userId", type="int", description="userId", required=false, in="query"),
	 *      @RequestParameter(name="start", type="int", description="start", required=false, in="query"),
	 *      @RequestParameter(name="limit", type="int", description="limit", required=false, in="query"),
	 *      @RequestParameter(name="filter", type="string", description="filter", required=false, in="query"),
	 *      @RequestParameter(name="sort", type="string", description="sort", required=false, in="query")
	 * })
	 */
	public function search(ApiRequest $request, ApiResponse $response): array
	{
		$query  = urldecode((string) $request->getParameter('query'));
		$offset = $request->getQueryParam('offset', 0);
		$limit  = $request->getQueryParam('limit', null);
		$filter = $request->getQueryParam('filter', null);
		$sort   = $request->getQueryParam('sort', null);
		$userId = $request->getQueryParam('userId', null);

		$qb = $this->prepareSearchQueryBuilder();

		if ($query) {
			$qb->andWhere(
				'pt.name LIKE :term OR p.code1 LIKE :term OR p.code2 LIKE :term OR p.ean LIKE :term OR p.id LIKE :term',
			)
				->setParameter('term', '%' . $query . '%');
		}

		$result              = $qb->getQuery()->getArrayResult();
		$canCacheProductsUrl = Config::load('canCacheProductsUrl');
		$useOldUrls          = Config::load('useOldUrls');
		$locale              = $this->translator->getLocale();
		foreach ($result as &$row) {
			$link = '#';
			if (!$canCacheProductsUrl) {
				$linkParams = [
					'id'     => $row['id'],
					'locale' => $locale,
				];

				//				if ($useOldUrls && $product->variantOf && $product->variantOf != $product->getId()) {
				//					$linkParams['variant'] = $product->getId();
				//				}

				//				$link = $this->linkGenerator->link('EshopCatalog:Front:Default:product', $linkParams); TODO @Roman trva to vygenerovat hrozne dlouho, proc?
			}
			$row['link'] = $link;

			if (!Validators::isNone($row['coverPath']) && !Validators::isNone($row['coverFilename'])) {
				$row['path']     = $row['coverPath'];
				$row['filename'] = $row['coverFilename'];
			}
			unset($row['coverPath'], $row['coverFilename']);
		}

		if (class_exists('EshopCheckout\DI\EshopCheckoutExtension') && Parameters::load(
				'eshopCheckout.allowPriceLevels',
				false,
			) && !is_null($userId)) {
			$this->loadPriceLevels($result, (int) $userId);
		}

		return $result;
	}

	protected function prepareSearchQueryBuilder(): QueryBuilder
	{
		// TODO do modelu
		/** @var QueryBuilder $qb */
		$qb = $this->productsDaoService->getEr()->createQueryBuilder('p', 'p.id')
			->select(
				'p.id, pt.name, p.price, p.code1, p.code2, p.ean, p.quantity, p.isPublished, pt.isPublished, vat.rate as vatRate, i.path, i.filename, cover.path as coverPath, cover.filename as coverFilename',
			)
			->join('p.productTexts', 'pt', 'WITH', 'pt.lang = :lang')
			->join('p.vatRate', 'vat')
			->leftJoin('p.gallery', 'g')
			->leftJoin('g.images', 'cover', Query\Expr\Join::WITH, 'cover.isCover = 1')
			->leftJoin('g.images', 'i', Query\Expr\Join::WITH, 'i.position = 0')
			->orderBy('pt.name')
			->setParameter('lang', $this->translator->getLocale());

		$qb->andWhere('p.isDeleted = 0 AND p.disableListing = 0');

		return $qb;
	}

	protected function loadPriceLevels(array &$products, int $userId = null): void
	{
		$daoProducts = [];
		foreach ($products as $product) {
			$dao = new Product($product['id']);
			$dao->setQuantity((int) $product['quantity']);
			$dao->setVatRate($product['vatRate']);
			$dao->setPrice((float) $product['price']);
			$dao->setBasePrice((float) $product['price']);
			$dao->priceInBaseCurrency     = $dao->getPrice();
			$dao->basePriceInBaseCurrency = $dao->basePrice;
			$daoProducts[$product['id']]  = $dao;
		}

		$this->productsDaoService->loadPriceLevels($daoProducts, $userId);

		foreach ($products as $k => $v) {
			$product               = $daoProducts[$v['id']];
			$products[$k]['price'] = $product->price;
		}
	}

}
