<?php declare(strict_types = 1);

namespace EshopCatalog\CronModule\Model;

use Contributte\Translation\Translator;
use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Images\ImagePipe;
use Core\Model\Sites;
use Currency\Model\Currencies;
use Currency\Model\Entities\Currency;
use Currency\Model\Exchange;
use Doctrine\ORM\Query\Expr\Join;
use EshopCatalog\FrontModule\Model\AvailabilityService;
use EshopCatalog\FrontModule\Model\Categories;
use EshopCatalog\FrontModule\Model\Dao\Product;
use EshopCatalog\FrontModule\Model\ProductsFacade;
use EshopCatalog\Model\Config;
use EshopCatalog\Model\Entities\Availability;
use EshopCatalog\Model\Entities\ProductPriceLevel;
use Exception;
use Gallery\FrontModule\Model\Dao\Image;
use Nette\Utils\FileSystem;
use Nette\Utils\Json;

class ProductsFeed
{
	public string  $baseUrl  = '';
	public ?string $generate = null;

	public function __construct(
		protected EntityManagerDecorator $em,
		protected ProductsFacade         $productsFacade,
		protected Categories             $categories,
		protected Sites                  $sites,
		protected AvailabilityService    $availabilityService,
		protected Exchange               $exchange,
		protected Currencies             $currencies,
		protected ImagePipe              $imagePipe,
		protected Translator             $translator,
	)
	{
	}

	/**
	 * @return array<int, int>
	 */
	protected function getAllSiteProductsId(): array
	{
		return $this->productsFacade->productsService->getProductsIdAll();
	}

	/**
	 * @return array{
	 *     availability: string,
	 *     availabilityIdent: string,
	 *     available: int,
	 * }
	 */
	protected function getAvailabilityData(Product $product): array
	{
		return [
			'availability'      => $product->getAvailability()->getName(),
			'availabilityIdent' => $product->getAvailability()->getIdent(),
			'available'         => $product->getQuantity() + $product->getQuantityExternal(),
		];
	}

	public function getProductsAvailability(): array
	{
		$data = [];
		$avs  = $this->availabilityService->getAll();

		$qb = $this->productsFacade->productsService->getEr()->createQueryBuilder('p')
			->select(
				'p.id, p.code1, p.ean, p.retailPrice, p.price, p.quantity, IDENTITY(p.availability) as availability, SUM(ps.quantity) as qeSum',
			)
			->innerJoin('p.sites', 's', Join::WITH, 's.site = :site AND s.isActive = 1')
			->leftJoin('p.suppliers', 'ps')
			->groupBy('p.id')
			->andWhere('p.isDeleted = 0')
			->andWhere('p.disableListing = 0')
			->setParameter('site', $this->sites->getCurrentSite()->getIdent());

		if (Config::load('product.publishedByLang')) {
			$qb->innerJoin('p.productTexts', 'pt', Join::WITH, 'pt.lang = :lang AND pt.isPublished = 1')
				->setParameter('lang', $this->translator->getLocale());
		} else {
			$qb->andWhere('p.isPublished = 1');
		}

		$soldOutAv = $this->availabilityService->getByIdent(Availability::SOLD_OUT);

		foreach ($qb->getQuery()->getScalarResult() as $row) {
			if (!$row['qeSum']) {
				$row['qeSum'] = 0;
			}

			$av = $avs[$row['availability']] ?: $soldOutAv;

			$item = [
				'id'                => $row['id'],
				'code1'             => $row['code1'],
				'ean'               => $row['ean'],
				'price'             => (float) $row['price'],
				'availability'      => $av ? $av->getName() : '',
				'availabilityIdent' => $av ? $av->getIdent() : '',
				'available'         => $row['quantity'] + $row['qeSum'],
			];

			if ($row['retailPrice'] && $row['retailPrice'] > $row['price']) {
				$item['retailPrice'] = (float) $row['retailPrice'];
			}

			$data[] = $item;
		}

		return $data;
	}

	public function getProducts(bool $returnData = false): array
	{
		$data = [
			'count' => 0,
		];

		/** @var Currency|null $czk */
		$czk = $this->currencies->getActive()['CZK'];

		/** @var Currency|null $eur */
		$eur = $this->currencies->getActive()['EUR'];

		$hasCzk = $czk ?? null;
		$hasEur = $eur ? (float) $eur->rate : null;

		$priceLevels = [];
		foreach ($this->em->createQueryBuilder()
			         ->select('IDENTITY(pl.productId) as product, IDENTITY(pl.groupId) as group, pl.price')
			         ->from(ProductPriceLevel::class, 'pl')
			         ->getQuery()->getArrayResult() as $row) {
			$priceLevels[$row['product']][$row['group']] = (float) $row['price'];
		}

		$currentSite   = $this->sites->getCurrentSite();
		$site          = $currentSite->getIdent();
		$baseFile      = WWW_DIR . '/feed-' . $site . '-' . $currentSite->getCurrentDomain()->getLang();
		$tmpFile       = $baseFile . '.tmp';
		$file          = $baseFile . '.json';
		$this->baseUrl = 'https://' . $currentSite->getCurrentDomain()->getDomain() . '/';

		FileSystem::delete($tmpFile);
		FileSystem::createDir(dirname($tmpFile));
		file_put_contents($tmpFile, '{');
		$isFirstRow = true;

		$this->em->getConfiguration()->setSQLLogger();
		foreach (array_chunk($this->getAllSiteProductsId(), 250) as $chunk) {
			foreach ($this->productsFacade->getProducts($chunk) as $product) {
				if (!$product->defaultCategoryId || !$product->getAvailability() || !$product->getAvailability()
						->canAddToCart()) {
					continue;
				}

				$link    = explode('/', $product->link);
				$link[2] = $currentSite->getCurrentDomain()->getDomain();
				$link    = implode('/', $link);

				$item = [
					'id'                  => $product->getId(),
					'code1'               => $product->getCode1(),
					'ean'                 => $product->getEan(),
					'title'               => $product->getName(),
					'description'         => strip_tags((string) $product->shortDescription),
					'longDescription'     => $product->getDescription(),
					'link'                => $link,
					'deliveryDate'        => $product->getAvailability()->getDelay(),
					'categoryId'          => [],
					'category'            => [],
					'categoryHierarchyId' => [],
					'categoryHierarchy'   => [],
					'tags'                => [],
					'vat'                 => $product->getVatRate(),
					'width'               => $product->width,
					'height'              => $product->height,
					'depth'               => $product->depth,
					'weight'              => $product->weight,
				];

				$usedCats = [];
				foreach (array_merge([$product->defaultCategoryId], $product->categories) as $catId) {
					$cat = $this->categories->get((int) $catId);

					if (!$cat || in_array($cat->getId(), $usedCats)) {
						continue;
					}

					$usedCats[]           = $cat->getId();
					$item['categoryId'][] = $cat->getId();
					$item['category'][]   = $cat->name;

					$parents = array_reverse($cat->getParentPath());
					$tmp1    = [];
					$tmp2    = [];
					foreach ($parents as $parent) {
						$usedCats[] = $parent->getId();
						$tmp1[]     = $parent->getId();
						$tmp2[]     = $parent->name;
					}

					$tmp1[] = $cat->getId();
					$tmp2[] = $cat->name;

					$item['categoryHierarchyId'][] = $tmp1;
					$item['categoryHierarchy'][]   = $tmp2;
				}

				if ($product->variantId) {
					$item['itemGroupId'] = $product->variantId;

					if ($product->variantOf == $product->getId()) {
						$item['isMaster'] = 1;
					} else {
						$item['variantOf'] = $product->variantOf;
					}
				}

				if ($product->getManufacturer()) {
					$item['brand'] = $product->getManufacturer()->name;
				}

				$addOriginalPrice = Config::load('product.allowRetailPrice') && $product->getRetailPrice(
					) && $product->getRetailPrice() > $product->getPrice();
				if ($hasCzk) {
					$item['priceCzk'] = $this->exchange->change($product->getPrice(), 'CZK');
					if ($addOriginalPrice) {
						$item['priceOriginalCzk'] = $this->exchange->change($product->getRetailPrice(), 'CZK');
					}
				}

				if ($hasEur !== null) {
					$item['rateEur'] = $hasEur;
				}

				foreach ($priceLevels[$product->getId()] ?? [] as $group => $price) {
					if ($hasCzk) {
						$item['priceLevels'][$group . '_czk'] = $this->exchange->change($price, 'CZK');
					}
				}

				$gallery = $product->getGallery();
				if ($gallery && $gallery->getCover()) {

					$item['imageLink'] = Config::load('productsFeed.makeThumbs')
						? $this->imagePipe->request(
							$gallery->getCover()
								->getFilePath(),
							Config::load('productsFeed.thumbsSize'),
							'fit',
						)
						: $this->baseUrl . ltrim($gallery->getCover()->getFilePath(), '/');

					$images = [];
					foreach ($gallery->getImages() as $image) {
						/** @var Image $image */
						$img = [
							'file' => Config::load('productsFeed.makeThumbs')
								? $this->imagePipe->request(
									$image->getFilePath(),
									Config::load('productsFeed.thumbsSize'),
									'fit',
								)
								: $this->baseUrl . ltrim($image->getFilePath(), '/'),
						];

						if (Config::load('productsFeed.makeThumbs')) {
							$img['fullFile'] = $this->baseUrl . ltrim($image->getFilePath(), '/');
						}

						foreach (['title', 'description', 'alt', 'source'] as $col) {
							if ($image->$col) {
								$img[$col] = $image->$col;
							}
						}

						if ($image->isCover) {
							array_unshift($images, $img);
						} else {
							$images[] = $img;
						}

						if ($image->id === $gallery->getCover()->id || isset($item['additionalImageLink'])) {
							continue;
						}

						$item['additionalImageLink'] = Config::load('productsFeed.makeThumbs')
							? $this->imagePipe->request(
								$image->getFilePath(),
								Config::load('productsFeed.thumbsSize'),
								'fit',
							)
							: $this->baseUrl . ltrim($image->getFilePath(), '/');
					}

					$item['images'] = $images;
				}

				$item += $this->getAvailabilityData($product);

				foreach ($product->getRelated() as $related) {
					$item['relatedProducts'][] = $related->getId();
				}

				if ($product->tags) {
					foreach ($product->tags as $tag) {
						$item['tags'][] = $tag->name;
					}
				}

				if ($product->isAssort) {
					$item['tags'][] = 'Assort';
				}

				if ($product->giftsIds) {
					$item['tags'][] = 'Gift';
					$item['gifts']  = $product->giftsIds;
				}

				if ($product->getFeatures()) {
					$item['features'] = [];
					foreach ($product->getFeatures() as $feature) {
						$item['features'][$feature->name] = $feature->value;
					}
				}

				file_put_contents(
					$tmpFile,
					(!$isFirstRow ? ',"' . $item['id'] . '":' : '"' . $item['id'] . '":') . Json::encode($item),
					FILE_APPEND,
				);
				$isFirstRow = false;
				$data['count']++;
			}
		}

		file_put_contents($tmpFile, '}', FILE_APPEND);
		FileSystem::rename($tmpFile, $file);
		FileSystem::delete($tmpFile);

		return $data;
	}

	public function loadAndParseProducts(): array
	{
		$currentSite = $this->sites->getCurrentSite();
		$site        = $currentSite->getIdent();
		$file        = WWW_DIR . '/feed-' . $site . '-' . $currentSite->getCurrentDomain()->getLang() . '.json';

		try {
			if (file_exists($file)) {
				return (array) Json::decode((string) file_get_contents($file));
			}
		} catch (Exception) {
		}

		return [];
	}

}
