<?php declare(strict_types = 1);

namespace EshopCatalog\CronModule\Presenters;

use Core\Model\Entities\EntityManagerDecorator;
use EshopCatalog\FrontModule\Model\CacheService;
use EshopCatalog\Model\Entities\Product;
use EshopCatalog\Model\Entities\ProductPriceHistory;
use Exception;
use Nette\Application\ForbiddenRequestException;
use Tracy\Debugger;

class MigrationsPresenter extends BasePresenter
{
	public function __construct(
		protected EntityManagerDecorator $em,
		protected CacheService           $cacheService,
	)
	{
		parent::__construct();
	}

	public function actionRetailPriceToHistory(): void
	{
		if (!$this->getUser()->isAllowed('Core:Admin', 'access')) {
			throw new ForbiddenRequestException();
		}

		set_time_limit(1000);

		try {
			$edited = [];
			$this->em->beginTransaction();
			$i = 0;
			foreach ($this->em->getRepository(Product::class)->createQueryBuilder('p')
				         ->select('p.id, p.retailPrice')
				         ->where('p.retailPrice > 0')
				         ->getQuery()->getArrayResult() as $row) {
				if (!$row['retailPrice']) {
					continue;
				}

				$edited[] = $row['id'];

				/** @var Product $product */
				$product = $this->em->getReference(Product::class, $row['id']);

				$history = new ProductPriceHistory(
					$product,
					(float) $row['retailPrice'],
					0.0,
					'retailPriceToHistory',
				);

				$this->em->persist($history);

				if ($i === 20) {
					$this->em->flush();
					$this->em->clear();
					$i = 0;
				}

				$i++;
			}

			$this->em->flush();
			$this->em->commit();

			foreach ($edited as $id) {
				foreach ($this->langsService->getLangs(false) as $lang) {
					$this->cacheService->productCache->remove('product/' . $id . '/' . $lang->getTag());
				}
			}
		} catch (Exception$e) {
			if ($this->em->getConnection()->isTransactionActive()) {
				$this->em->rollback();
			}

			Debugger::log($e, 'retailPriceToHistory');
			$this->sendJson([
				'result'  => 'error',
				'message' => $e->getMessage(),
			]);
		}

		$this->sendJson([
			'result' => 'ok',
		]);
	}
}
