<?php declare(strict_types = 1);

namespace EshopCatalog\FrontModule\Model\Dao;

use Core\Model\Dao\Traits\ExtraFields;
use Core\Model\Entities\TSeo;
use DateTimeInterface;

class Category
{
	use TSeo;
	use ExtraFields;

	public ?int    $id               = null;
	public ?string $name             = null;
	public ?string $nameH1           = null;
	public ?string $alias            = null;
	public ?string $shortDescription = null;
	public ?string $description      = null;
	public ?string $emptyText        = null;
	public ?string $image            = null;
	public ?int    $lvl              = null;

	/** @var self[] */
	public array $child = [];

	public ?int  $parentId = null;
	public ?self $parent   = null;

	public ?string            $link              = '#';
	public ?DateTimeInterface $created           = null;
	public ?DateTimeInterface $modified          = null;
	public int                $filtersFromParent = 0;
	public array               $attrs                = [];
	public ?string             $rod                  = null;
	public bool                $canProductsAddToCart = true;

	/** @var self[] */
	public array $related = [];

	public function getId(): int { return $this->id; }

	public function getNameH1(): string { return $this->nameH1 ?: $this->name; }

	public function setParent(Category $category): self
	{
		$this->parent = &$category;

		return $this;
	}

	public function getParent(): ?Category { return $this->parent; }

	/**
	 * @return self[]
	 */
	public function getParentPath(): array
	{
		$path = [];

		$parent = $this->getParent();
		while ($parent) {
			$path[$parent->getId()] = $parent;
			$parent                 = $parent->getParent();
		}

		return $path;
	}

	public function getParentPathString(): string
	{
		return implode(' > ', array_map(static fn($cat) => $cat->name, $this->getParentPath()));
	}

	public function getParentPathStringFlipped(): string
	{
		return implode(' > ', array_map(static fn($cat) => $cat->name, array_reverse($this->getParentPath())));
	}

	public function setParentId(int $id): self
	{
		$this->parentId = $id;

		return $this;
	}

	public function getParentId(): ?int { return $this->parentId; }

	public function setChild(array $arr): self
	{
		$this->child = $arr;

		return $this;
	}

	public function getRoot(): ?Category
	{
		$r = $this->getParent();

		while ($r && $r->getParent()) {
			$r = $r->getParent();
		}

		return $r;
	}

	public function getRootId(): ?int
	{
		$cat = $this;
		while ($cat) {
			if (!$cat->getParent()) {
				return $cat->getParentId();
			}
			$cat = $cat->getParent();
		}

		/** @phpstan-ignore-next-line */
		return $this->getParentId();
	}

	/**
	 * @return Category[]
	 */
	public function getChild(): array { return $this->child ?: []; }

	public function addAttr(string $key, mixed $value): self
	{
		$this->attrs[$key] = $value;

		return $this;
	}

	public function setAttrs(array $attrs): self
	{
		$this->attrs = $attrs;

		return $this;
	}

	/**
	 * @return mixed|null
	 */
	public function getAttr(string $key) { return $this->attrs[$key] ?? null; }

	public function getAttrs(): array { return $this->attrs ?: []; }

}
