<?php declare(strict_types = 1);

namespace EshopCatalog\FrontModule\Model;

use Core\Model\Entities\EntityManagerDecorator;
use Currency\Model\Exchange;
use Doctrine\ORM\Query\Expr\Join;
use EshopCatalog\FrontModule\Model\Dao\FilterGroup;
use EshopCatalog\FrontModule\Model\Dao\FilterItem;
use EshopCatalog\Model\Config;
use EshopCatalog\Model\Entities\CategoryFilter;
use EshopCatalog\Model\Entities\Feature;
use EshopCatalog\Model\Entities\FeatureProduct;
use EshopCatalog\Model\Entities\FeatureValue;
use EshopCatalog\Model\Entities\Product;
use Exception;
use Nette\Caching\Cache;
use Nette\SmartObject;
use Nette\Utils\DateTime;

/**
 * TODO misto hledání podle productsId používat třeba kategorie
 */
class FilterService
{
	use SmartObject;

	public static array $prepareDataCategories = [];
	public static array $cacheTags             = [];
	final public const CACHE_NAMESPACE = 'eshopCatalogFilter';
	protected ?array $cProductsData = null;
	protected array  $cacheDep      = [
		Cache::Tags   => ['filters'],
		Cache::Expire => '10 minutes',
	];

	protected ?array $cFeatureProducts = null;
	protected ?array $cProducts        = null;

	public function __construct(
		protected EntityManagerDecorator $em,
		protected CacheService           $cacheService,
		protected Manufacturers          $manufacturers,
		protected Features               $featuresService,
		protected Config                 $config,
		protected Tags                   $tags,
		protected Exchange               $exchange,
	)
	{
	}

	/**
	 * @param int[] $productIds
	 */
	public function getFeatures(array $productIds = []): array
	{
		$groups = [];
		$qb     = $this->em->getRepository(FeatureProduct::class)->createQueryBuilder('fp')
			->select('IDENTITY(fv.feature) as featureId, IDENTITY(fp.featureValue) as valueId')
			->join('fp.featureValue', 'fv', 'WITH', 'fv.isPublished = 1')
			->join('fv.feature', 'f', 'WITH', 'f.useAsFilter = 1')
			->orderBy('f.position')->addOrderBy('fv.position');

		if ($productIds) {
			$qb->where('fp.product IN (' . implode(',', array_values($productIds)) . ')');
		}

		$data = $qb->getQuery()->getScalarResult();

		foreach ($data as $row) {
			if (!isset($groups[$row['featureId']])) {
				$tmp = $this->featuresService->getFeatureById($row['featureId']);

				if ($tmp && $tmp['name']) {
					$tmpV            = new FilterGroup;
					$tmpV->id        = $tmp['id'];
					$tmpV->name      = $tmp['name'];
					$tmpV->type      = 'feature';
					$tmpV->valueType = $tmp['type'];
					$tmpV->unit      = $tmp['unit'] ? (string) $tmp['unit'] : null;
					$tmpV->position  = (int) $tmp['position'];

					$groups[$row['featureId']] = $tmpV;
				}
			}

			$g = $groups[$row['featureId']];

			if (!isset($g->items[$row['valueId']])) {
				$tmp = $this->featuresService->getFeatureValueById($row['valueId']);

				if ($tmp && $tmp['name']) {
					$tmpV        = new FilterItem;
					$tmpV->id    = $tmp['id'];
					$tmpV->value = $tmp['name'];
					$tmpV->type  = 'featureValue';
					$tmpV->rodM  = $tmp['rodM'];
					$tmpV->rodZ  = $tmp['rodZ'];
					$tmpV->rodS  = $tmp['rodS'];

					$g->items[$row['valueId']] = $tmpV;
				}
			}
		}

		return $groups;
	}

	/**
	 * @param int[] $productIds
	 *
	 * @return FilterItem[]
	 */
	public function getManufacturers($productIds): array
	{
		$toUppercase = $this->config->get('filterManufacturersUppercase', false);
		$data        = [];

		$tmp = array_intersect_key(
			$this->manufacturers->getManufacturers(),
			$this->getProductsData($productIds)['m'] ?? [],
		);
		uasort($tmp, static fn($a, $b) => $a->name <=> $b->name);

		foreach ($tmp as $m) {
			$data[$m->id]        = new FilterItem;
			$data[$m->id]->id    = $m->id;
			$data[$m->id]->value = $toUppercase ? mb_strtoupper($m->name) : $m->name;
		}

		return $data;
	}

	/**
	 * @param int[] $productIds
	 *
	 * @return FilterItem[]
	 */
	public function getTags(array $productIds): array
	{
		$data = [];

		$tmp = array_intersect_key($this->tags->getAll(), $this->getProductsData($productIds)['t'] ?? []);
		uasort($tmp, static fn($a, $b) => $a->name <=> $b->name);

		foreach ($tmp as $tag) {
			$data[$tag->id]        = new FilterItem;
			$data[$tag->id]->id    = $tag->id;
			$data[$tag->id]->value = $tag->name;

			$data[$tag->id]->customData['tagColor'] = [
				'text' => $tag->textColor,
				'bg'   => $tag->bgColor,
			];
		}

		return $data;
	}

	/**
	 * @return FilterGroup[]
	 */
	public function getCategoryFilters(int $id, bool $deep = true): array
	{
		$key  = 'categoryFilters_' . $id . ($deep ? '_deep' : '');
		$data = [];

		$features = $this->cacheService->categoryCache->load($key, function(&$dep) use ($id, $deep) {
			$dep                = $this->cacheDep;
			$dep[Cache::Expire] = '14 days';
			$dep[Cache::Tags][] = 'category/' . $id;

			$find = fn(int $id) => $this->em->getRepository(CategoryFilter::class)->createQueryBuilder('f')
				->select('IDENTITY(f.feature) as featureId')
				->andWhere('f.category = :category')->setParameters(['category' => $id])
				->orderBy('f.position')
				->getQuery()->getArrayResult();

			$features = $find($id);

			if (!$features && $deep) {
				try {
					$stmt      = $this->em->getConnection()->prepare(
						"
						SELECT T2.id 
						FROM (
							SELECT @r AS _id,
								(SELECT @r := parent_id FROM eshop_catalog__category WHERE id = _id) AS parent_id, @l := @l + 1 AS lvl 
								FROM (SELECT @r := {$id}, @l := 0) vars, eshop_catalog__category h WHERE @r <> 0) T1 
						JOIN eshop_catalog__category T2 ON T1._id = T2.id ORDER BY T1.lvl DESC
						",
					);
					$resultSet = $stmt->executeQuery();
					$tmp       = $resultSet->fetchAllAssociative();
				} catch (Exception) {
					$tmp = [];
				}
				array_shift($tmp);
				array_pop($tmp);
				$categories = [];
				foreach ($tmp as $row) {
					$dep[Cache::Tags][] = 'category/' . $row['id'];
					$categories[]       = $row['id'];
				}

				/** @phpstan-ignore-next-line */
				while (!empty($categories) && empty($features)) {
					$id       = array_pop($categories);
					$features = $find((int) $id);

					if ($features) {
						break;
					}
				}
			}

			return array_map(static fn($row) => $row['featureId'], $features);
		});

		if (!$features) {
			return $data;
		}

		foreach ($features as $featureId) {
			$tmp = $this->featuresService->getFeatureById($featureId);

			$group            = new FilterGroup;
			$group->id        = (int) $tmp['id'];
			$group->name      = $tmp['name'] ? (string) $tmp['name'] : null;
			$group->type      = 'feature';
			$group->valueType = $tmp['type'];
			$group->unit      = $tmp['unit'] ? (string) $tmp['unit'] : null;
			$group->position  = (int) $tmp['position'];
			$group->decimals  = (int) $tmp['decimals'] ?: 0;

			$data[$featureId] = $group;

			if ($group->valueType !== Feature::TYPE_RANGE) {
				foreach ($this->featuresService->getValuesByFeatureId((int) $featureId) as $tmp) {
					$item        = new FilterItem;
					$item->id    = (int) $tmp['id'];
					$item->value = $tmp['name'];
					$item->type  = 'featureValue';

					if ($tmp['valueType']) {
						$item->valueType = $tmp['valueType'];
					}

					if ($item->valueType === Feature::VALUE_TYPE_COLOR) {
						$item->customData['colors']      = $tmp['extraFields']['colors'];
						$item->customData['colorsStyle'] = $tmp['extraFields']['colorsStyle'];
					}

					$data[$featureId]->items[$tmp['id']] = $item;
				}
			}
		}

		return $data;
	}

	protected function prepareData(array $productsIds): array
	{
		$catIds = !empty(self::$prepareDataCategories) ? implode(',', self::$prepareDataCategories) : null;

		if ($catIds) {
			$cacheKey = 'filtersPrepareData_' . md5($catIds);

			$result = $this->cacheService->filterCache->load($cacheKey);

			if ($result) {
				return $result;
			}
		}

		$data = [];
		$now  = (new DateTime)->format('Y-m-d H:i:00');

		foreach (array_chunk($productsIds, 999) as $chunk) {
			$featureValuesParent = [];
			foreach ($this->em->createQueryBuilder()->select('IDENTITY(fv.feature) as feature, fv.id')
				         ->from(FeatureValue::class, 'fv')
				         ->innerJoin(
					         FeatureProduct::class,
					         'fp',
					         Join::WITH,
					         'fp.featureValue = fv.id AND fp.product IN (' . implode(',', $chunk) . ')',
				         )
				         ->getQuery()->getArrayResult() as $v) {
				$featureValuesParent[$v['id']] = $v['feature'];
			}

			$qb = $this->em->createQueryBuilder()
				->select('p.id as id, p.price as price, IDENTITY(p.availability) as av, IDENTITY(p.manufacturer) as m')
				->from(Product::class, 'p')
				->innerJoin('p.vatRate', 'vr')
				->addSelect('vr.rate as vatRate')
				->leftJoin('p.featureProducts', 'fp')
				->addSelect('GROUP_CONCAT(DISTINCT IDENTITY(fp.featureValue)) as featureValues')
				->leftJoin('p.dynamicFeatures', 'dfp')
				->addSelect(
					'GROUP_CONCAT(DISTINCT CONCAT(IDENTITY(dfp.feature),\'|\',dfp.value) SEPARATOR \';\') as dynamicFeatures',
				)
				->leftJoin(
					'p.productTags',
					'pt',
					Join::WITH,
					'(pt.validFrom IS NULL OR pt.validFrom <= :now) AND (pt.validTo IS NULL OR pt.validTo >= :now)',
				)
				->addSelect('GROUP_CONCAT(DISTINCT IDENTITY(pt.tag)) as tags')
				->andWhere('p.id IN (' . implode(',', $chunk) . ')')
				->setParameters([
					'now' => $now,
				])
				->groupBy('p.id');
			foreach ($qb->getQuery()->getArrayResult() as $v) {
				$df       = [];
				$features = [];
				if ($v['dynamicFeatures']) {
					foreach (explode(';', (string) $v['dynamicFeatures']) as $kv) {
						$kv = explode('|', $kv);
						if (count($kv) === 2) {
							$df[(int) $kv[0]][] = $kv[1];
						}
					}
				}

				if ($v['featureValues']) {
					foreach (explode(',', (string) $v['featureValues']) as $v2) {
						$features[$featureValuesParent[$v2]][] = (int) $v2;
					}
				}

				$id      = (int) $v['id'];
				$price   = (float) $v['price'];
				$vatRate = (int) $v['vatRate'];
				$tags    = $v['tags'] ? array_map(static fn($v2) => (int) $v2, explode(',', (string) $v['tags'])) : [];
				$manu    = $v['m'] ? (int) $v['m'] : null;

				$data['p'][$id] = [
					'id'      => $id,
					'price'   => $price,
					'vatRate' => $vatRate,
					'av'      => $v['av'],
					'df'      => $df,
				];

				foreach ($features as $k2 => $v2) {
					foreach ($v2 as $v3) {
						$data['filters']['feature_' . $v3][] = $id;
					}

					$data['p'][$id]['filters']['feature_' . $k2] = $v2;
				}

				foreach ($tags as $t) {
					$data['t'][$t][]                    = $id;
					$data['filters']['tag_' . $t][]     = $id;
					$data['p'][$id]['filters']['tag_0'] = $t;
				}

				$data['m'][(int) $manu][$id]              = $id;
				$data['filters']['manu_' . (int) $manu][] = $id;
				$data['p'][$id]['filters']['manu_0']      = $manu;

				if (!isset($data['priceMin']) || $data['priceMin'] > $price) {
					$data['priceMin']        = $price;
					$data['priceMinVatRate'] = $vatRate;
				}
				if (!isset($data['priceMax']) || $data['priceMax'] < $price) {
					$data['priceMax']        = $price;
					$data['priceMaxVatRate'] = $vatRate;
				}

				foreach ($df as $dfK => $dfV) {
					foreach ($dfV as $dfV2) {
						if (!isset($data['df'][$dfK]['min']) || $data['df'][$dfK]['min'] > $dfV2) {
							$data['df'][$dfK]['min'] = $dfV2;
						}
						if (!isset($data['df'][$dfK]['max']) || $data['df'][$dfK]['max'] < $dfV2) {
							$data['df'][$dfK]['max'] = $dfV2;
						}
					}
				}
			}
		}

		if ($catIds) {
			$this->cacheService->filterCache->save($cacheKey, $data, $this->cacheDep);
		}

		return $data;
	}

	public function getProductsData(array $productsIds, string $cacheKey = ''): ?array
	{
		if (!$this->cProductsData) {
			$this->cProductsData = $this->prepareData($productsIds);
		}

		return $this->cProductsData;
	}

}
