<?php declare(strict_types = 1);

namespace EshopCatalog\FrontModule\Model;

use Core\Model\Entities\EntityManagerDecorator;
use Doctrine\ORM\Query;
use Doctrine\ORM\Query\Expr\Join;
use EshopCatalog\Model\Entities\ProductDocument;
use EshopOrders\Model\Entities\Order;

class OrderDocuments
{
	public function __construct(protected EntityManagerDecorator $em)
	{
	}

	public function find(int $documentId, int $orderId, int $productId): ?ProductDocument
	{
		/** @var ProductDocument|null $document */
		$document = $this->em->getRepository(ProductDocument::class)->createQueryBuilder('d')
			->where('d.id = :document')
			->innerJoin('d.product', 'product', Join::WITH, 'product.id = :product')
			->setParameters([
				'document' => $documentId,
				'product'  => $productId,
			])
			->setMaxResults(1)->getQuery()
			->setHint(Query::HINT_FORCE_PARTIAL_LOAD, true)->getOneOrNullResult();

		$status = $this->em->getRepository(Order::class)->createQueryBuilder('o')
			->select('o.isPaid')
			->where('o.id = :order')
			->innerJoin('o.orderItems', 'oi', Join::WITH, 'oi.product = :product')
			->setParameters([
				'order'   => $orderId,
				'product' => $productId,
			])->setMaxResults(1)->getQuery()->getArrayResult()[0] ?? null;

		if (!$document || !$status) {
			return null;
		}

		if ($document->isSecured() && $status['isPaid'] == 0) {
			return null;
		}

		return $document;
	}

}
