<?php declare(strict_types = 1);

namespace EshopCatalog\FrontModule\Model\Subscribers;

use Contributte\Translation\Translator;
use Core\Model\Event\ControlEvent;
use Core\Model\UI\FrontPresenter;
use EshopCatalog\FrontModule\Model\ProductsFacade;
use EshopCatalog\Model\Entities\Availability;
use EshopOrders\FrontModule\Model\Carts;
use EshopOrders\FrontModule\Model\Event\AddedCartItemEvent;
use EshopOrders\FrontModule\Presenters\DefaultPresenter;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

class CartSubscriber implements EventSubscriberInterface
{
	protected bool $cannotAddProduct = false;

	public function __construct(
		protected Carts          $cartsService,
		protected Translator     $translator,
		protected ProductsFacade $productsFacade,
	)
	{
	}

	public static function getSubscribedEvents(): array
	{
		return [
			FrontPresenter::class . '::beforeRender'   => 'beforeRender',
			'eshopOrders.cartAddItem'                  => ['onAddItem', 200],
			DefaultPresenter::class . '::beforeRender' => 'defaultBeforeRender',
		];
	}

	public function defaultBeforeRender(ControlEvent $event): void
	{
	}

	public function beforeRender(ControlEvent $event): void
	{
		if (!$this->cannotAddProduct) {
			return;
		}

		$event->control->getPresenter()->payload->createModalData = [
			'title' => $this->translator->translate('eshopCatalogFront.cart.cannotAddProductToCart'),
			'text'  => $this->translator->translate('eshopCatalogFront.cart.cannotCombinePreorderAndStockProduct'),
		];
		$event->control->getPresenter()->payload->cartEvent       = null;
		$event->control->getPresenter()->sendPayload();
	}

	public function onAddItem(AddedCartItemEvent $event): void
	{
		$cart = $this->cartsService->getCurrentCart();

		if (!$cart || !$event->item->productId || $cart->getItemsCount() === 0) {
			return;
		}

		$product = $this->productsFacade->getProduct($event->item->productId);

		$hasPreorderProduct = $cart->hasPreorderProduct();
		$productIsPreorder  = $product && $product->getAvailability()->getIdent() === Availability::PREORDER;

		if ($hasPreorderProduct !== $productIsPreorder) {
			$this->cannotAddProduct = true;
			$event->stopPropagation();
		}
	}

}
