<?php declare(strict_types = 1);

namespace EshopCatalog\FrontModule\Model\Subscribers;

use Core\Model\Entities\EntityManagerDecorator;
use EshopCatalog\Model\Config;
use EshopCatalog\Model\PseudoWarehouse;
use EshopOrders\FrontModule\Model\Event\OrderEvent;
use EshopOrders\Model\Entities\OrderItem;
use Nette\Utils\Validators;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

class OrderPseudoWarehouseSubscriber implements EventSubscriberInterface
{
	/** @var OrderItem[] */
	protected array $beforeOrderItemsChange = [];
	protected array $changes                = [];

	public function __construct(
		protected PseudoWarehouse        $pseudoWarehouse,
		protected EntityManagerDecorator $em,
	)
	{
	}

	public static function getSubscribedEvents(): array
	{
		return [
			'eshopOrders.orderOnSuccess'          => ['orderOnSuccess', 100],
			'eshopOrders.admin.beforeOrderChange' => ['beforeOrderChange', 100],
			'eshopOrders.admin.orderOnSuccess'    => ['orderOnSuccess', 100],
			'eshopCheckout.orderBeforeSave'       => ['orderBeforeSave', 100],
		];
	}

	public function beforeOrderChange(OrderEvent $event): void
	{
		if ($event->order->getId() !== null) {
			$this->beforeOrderItemsChange = $event->order->getOrderItems()->toArray();
		}
	}

	public function orderBeforeSave(OrderEvent $event): void
	{
		if ($event->order->getId() !== null) {
			foreach ($event->order->getOrderItems() as $orderItem) {
				if ($orderItem->getId() !== null) {
					$quantity                                  = (int) $this->em->getConnection()->fetchOne('SELECT quantity FROM eshop_orders__order_item WHERE id = ?', [$orderItem->getId()]);
					$this->changes[$orderItem->getProductId()] = $quantity - $orderItem->getQuantity();
				} else {
					$this->changes[$orderItem->getProductId()] = -$orderItem->getQuantity();
				}
			}

			$removedOrderItemIds = array_diff(
				array_filter(array_map(static fn(OrderItem $oi) => $oi->getId(), $this->beforeOrderItemsChange), static fn($val) => !Validators::isNone($val)),
				array_filter(array_map(static fn(OrderItem $oi) => $oi->getId(), $event->order->getOrderItems()->toArray()), static fn($val) => !Validators::isNone($val))
			);
			foreach ($this->beforeOrderItemsChange as $oi) {
				if (in_array($oi->getId(), $removedOrderItemIds)) {
					$this->changes[$oi->getProductId()] = $oi->getQuantity();
				}
			}
		}
	}

	public function orderOnSuccess(OrderEvent $event): void
	{
		if (!Config::load('pseudoWarehouse')) {
			return;
		}

		$data = [];

		if (count($this->changes) > 0) {
			$data = $this->changes;
		} else {
			foreach ($event->order->getOrderItems() as $v) {
				if ($v->getProductId()) {
					$data[$v->getProductId()] = -$v->getQuantity();
				}
			}
		}

		foreach ($event->order->getGifts() as $v) {
			if (!$v->getProduct()) {
				continue;
			}
			$prodId = $v->getProduct()->getId();

			if (isset($data[$prodId])) {
				$data[$prodId] -= 1;
			} else {
				$data[$prodId] = -1;
			}
		}

		$lowerQuantities = [];
		$raiseQuantities = [];
		foreach ($data as $prodId => $q) {
			if ($q <= 0) {
				$lowerQuantities[$prodId] = abs($q);
			} else {
				$raiseQuantities[$prodId] = $q;
			}
		}

		if ($lowerQuantities) {
			$this->pseudoWarehouse->lowerQuantities($lowerQuantities, $event->order);
		}

		if ($raiseQuantities) {
			$this->pseudoWarehouse->raiseQuantities($data, $event->order);
		}
	}
}
