<?php declare(strict_types = 1);

namespace EshopCatalog\FrontModule\Model;

use Core\Model\Helpers\BaseFrontEntityService;
use Core\Model\Helpers\DaoHelper;
use Core\Model\Sites;
use Doctrine\ORM\Query;
use EshopCatalog\FrontModule\Model\Dao;
use EshopCatalog\FrontModule\Model\Dao\Product;
use EshopCatalog\Model\Config;
use EshopCatalog\Model\Entities\ProductTag;
use EshopCatalog\Model\Entities\Tag;
use EshopOrders\Model\Entities\PaymentSpedition;
use EshopOrders\Model\Entities\Spedition;
use EshopOrders\Model\Helpers\EshopOrdersCache;
use Nette\Caching\Cache;
use Nette\Utils\DateTime;
use Throwable;

class Tags extends BaseFrontEntityService
{
	final public const CACHE_NAMESPACE = 'eshopCatalogTags';

	protected $entityClass = Tag::class;

	protected ?Cache $productsCache    = null;
	protected ?array $cTags            = null;
	protected ?float $cFreeFrom        = null;
	protected ?array $cTagsForProducts = null;
	protected ?array $freeFromArr      = null;
	protected array  $cacheDep         = [
		Cache::Tags   => [self::CACHE_NAMESPACE, Products::CACHE_NAMESPACE],
		Cache::Expire => '1 week',
	];

	public function __construct(protected Sites $sites, protected EshopOrdersCache $eshopOrdersCache)
	{
	}

	public function getCache(): Cache
	{
		if ($this->cache === null) {
			$this->cache = new Cache($this->cacheStorage, self::CACHE_NAMESPACE);
		}

		return $this->cache;
	}

	public function getProductsCache(): Cache
	{
		if ($this->productsCache === null) {
			$this->productsCache = new Cache($this->cacheStorage, Products::CACHE_NAMESPACE);
		}

		return $this->productsCache;
	}

	/**
	 * @param Product|Product[] $product
	 *
	 * @throws Throwable
	 */
	public function loadTagsToProduct(Product|array &$product): void
	{
		$products = is_array($product) ? $product : [$product];

		$tags = $this->getIdsForProducts();

		$tagForPrice  = Config::load('tagWhenPriceIsLowerThenRetail');
		$freeDelivery = $this->get('freeDelivery');

		foreach ($products as &$prod) {
			$freeDeliveryMinimum = $this->getFreeDeliveryMinimumPrice($prod->disabledSpeditions ?: []);

			if (isset($tags[$prod->getId()])) {
				foreach ($tags[$prod->getId()] as $tagId) {
					$prod->tags[$tagId] = $this->get($tagId);
				}
			}

			if (Config::load('product.allowRetailPrice') && $tagForPrice && $prod->getPrice() < $prod->getRetailPrice(
				)) {
				$prod->tags[$tagForPrice] = $this->get($tagForPrice);
			}

			if ($freeDeliveryMinimum !== null && $freeDelivery && $prod->getPrice() >= $freeDeliveryMinimum) {
				$freeDelivery->isAuto       = true;
				$prod->tags['freeDelivery'] = $freeDelivery;
			}
		}
	}

	public function getIdsForProducts(): array
	{
		if ($this->cTagsForProducts === null) {
			$this->cTagsForProducts = [];
			$now                    = (new DateTime)->format('Y-m-d H:i:00');
			$allTags                = $this->getAll();
			$key                    = 'idsForProducts';

			$this->cTagsForProducts = $this->getCache()->load($key, function(&$dep) use ($now, $allTags) {
				$dep = [
					Cache::Tags       => [self::CACHE_NAMESPACE],
					Cache::Expire => '15 minutes',
				];

				$data = [];
				foreach ($this->em->createQueryBuilder()
					         ->select('IDENTITY(pt.product) as product, IDENTITY(pt.tag) as tag')
					         ->from(ProductTag::class, 'pt')
					         ->andWhere('pt.validFrom IS NULL OR pt.validFrom <= :now')
					         ->andWhere('pt.validTo IS NULL OR pt.validTo >= :now')
					         ->setParameter('now', $now)
					         ->getQuery()
					         ->getArrayResult() as $row) {
					$data[$row['product']][] = $allTags[$row['tag']]->type;
				}

				return $data;
			});
		}

		return $this->cTagsForProducts;
	}

	/**
	 * @throws Throwable
	 */
	public function get(int|string $id): ?Dao\Tag
	{
		return $this->getAll()[$id] ?? null;
	}

	/**
	 * @return Dao\Tag[]
	 * @throws Throwable
	 */
	public function getAll(): array
	{
		if ($this->cTags === null) {
			$lang        = $this->translator->getLocale();
			$this->cTags = $this->getCache()->load('allTags/' . $lang, function(&$dep) use ($lang) {
				$dep  = [
					Cache::Tags       => [self::CACHE_NAMESPACE],
					Cache::Expire => '1 month',
				];
				$data = [];

				foreach ($this->getEr()->createQueryBuilder('t')
					         ->select('t.id, t.type, t.image, tt.name, t.color, t.bgColor')
					         ->join('t.texts', 'tt', 'WITH', 'tt.lang = :lang')
					         ->setParameter('lang', $lang)
					         ->getQuery()
					         ->getArrayResult() as $row) {
					$tag                = $this->fillDao($row);
					$data[$row['type']] = $tag;
					$data[$row['id']]   = $tag;
				}

				return $data;
			});
		}

		return $this->cTags ?? [];
	}

	protected function fillDao(array $tag): Dao\Tag
	{
		/** @var Dao\Tag $dao */
		$dao            = DaoHelper::fillDaoFromArray($tag, new Dao\Tag);
		$dao->textColor = $tag['color'];
		$dao->isAuto    = false;

		return $dao;
	}

	protected function loadFreeFromArr(): array
	{
		if ($this->freeFromArr === null) {
			$siteIdent = $this->sites->getCurrentSite()->getIdent();

			$this->freeFromArr = $this->eshopOrdersCache->getCache()->load('freeFrom_' . $siteIdent, function(&$dep) {
				$dep = [
					Cache::Tags       => ['freeFrom', 'spedition', 'payment'],
					Cache::Expire => '1 week',
				];

				$arr = [];

				foreach ($this->em->getRepository(Spedition::class)->createQueryBuilder('s')
					         ->select('s.id, s.freeFrom')
					         ->innerJoin(PaymentSpedition::class, 'ps', Query\Expr\Join::WITH, 'ps.spedition = s.id')
					         ->innerJoin('ps.sites', 'sites', Query\Expr\Join::WITH, 'sites.ident = :ident')
					         ->where('s.freeFrom > 0')->andWhere('s.freeFrom IS NOT NULL')
					         ->setParameter('ident', $this->sites->getCurrentSite()->getIdent())
					         ->groupBy('s.id')
					         ->getQuery()->getArrayResult() as $row) {
					$arr[$row['id']] = (float) $row['freeFrom'];
				}

				return $arr;
			});
		}

		return $this->freeFromArr;
	}

	protected function getFreeDeliveryMinimumPrice(array $skipSpeditions = []): ?float
	{
		if (!empty($skipSpeditions)) {
			$min            = 99_999_999;
			$skipSpeditions = array_flip($skipSpeditions);
			foreach ($this->loadFreeFromArr() as $id => $val) {
				if (isset($skipSpeditions[$id])) {
					continue;
				}

				if ($val < $min) {
					$min = $val;
				}
			}

			return $min;
		}

		if ($this->cFreeFrom === null) {
			if ($this->loadFreeFromArr()) {
				$this->cFreeFrom = min($this->loadFreeFromArr());
			}
		}

		return $this->cFreeFrom;
	}

}
