<?php declare(strict_types = 1);

namespace EshopCatalog\Model\Entities;

use Core\Model\Entities\Repository\NestedTreeRepository;
use Core\Model\Entities\TId;
use Core\Model\Entities\TranslateListener;
use Core\Model\Entities\TTranslateListener;
use Core\Model\Helpers\Traits\TExtraField;
use DateTimeInterface;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\DBAL\Types\Types;
use Doctrine\ORM\Mapping as ORM;
use Gedmo\Mapping\Annotation as Gedmo;
use Nette\Utils\ArrayHash;
use Nette\Utils\DateTime;

#[ORM\Table(name: 'eshop_catalog__category')]
#[ORM\Index(columns: ['is_published', 'id', 'parent_id', 'lft'], name: 'publish_id_id_lft')]
#[ORM\Index(columns: ['tree_root', 'lft', 'id'], name: 'root_lft_id')]
#[Gedmo\Tree(type: 'nested')]
#[ORM\Entity(repositoryClass: NestedTreeRepository::class)]
#[ORM\EntityListeners([TranslateListener::class, CategoryListener::class])]
class Category
{
	use TId;
	use TTranslateListener;
	use TExtraField;

	public const EXTRA_FIELD_SECTION = 'eshopCatalogCategory';

	#[ORM\Column(name: 'image', type: Types::STRING, length: 255, nullable: true)]
	public ?string $image = null;

	#[ORM\Column(type: Types::SMALLINT, options: ['default' => 0])]
	public int $isPublished = 0;

	/**
	 * @var Collection<Category>
	 */
	#[ORM\OneToMany(mappedBy: 'parent', targetEntity: Category::class)]
	#[ORM\OrderBy(['lft' => 'ASC'])]
	public Collection $children;

	#[Gedmo\TreeParent]
	#[ORM\ManyToOne(targetEntity: Category::class, cascade: ['persist'], inversedBy: 'childrens')]
	#[ORM\JoinColumn(name: 'parent_id', referencedColumnName: 'id', onDelete: 'CASCADE')]
	public ?Category $parent = null;

	#[Gedmo\TreeLeft]
	#[ORM\Column(type: Types::INTEGER)]
	private int $lft;

	#[Gedmo\TreeLevel]
	#[ORM\Column(type: Types::INTEGER)]
	protected int $lvl;

	#[Gedmo\TreeRight]
	#[ORM\Column(name: 'rgt', type: Types::INTEGER)]
	private int $gt;

	#[Gedmo\TreeRoot]
	#[ORM\ManyToOne(targetEntity: Category::class)]
	#[ORM\JoinColumn(name: 'tree_root', referencedColumnName: 'id', onDelete: 'CASCADE')]
	private ?Category $root = null;

	/**
	 * @var Collection<CategoryTexts>
	 */
	#[ORM\OneToMany(mappedBy: 'id', targetEntity: CategoryTexts::class, cascade: ['persist', 'remove'], indexBy: 'lang')]
	protected Collection $categoryTexts;

	/**
	 * @var Collection<int, CategoryProduct>
	 */
	#[ORM\OneToMany(mappedBy: 'category', targetEntity: CategoryProduct::class, cascade: ['all'], indexBy: 'category')]
	protected Collection $categoryProducts;

	/**
	 * @var Collection<CategoryFilter>
	 */
	#[ORM\OneToMany(mappedBy: 'category', targetEntity: CategoryFilter::class, indexBy: 'feature_id')]
	#[ORM\OrderBy(['position' => 'ASC'])]
	public Collection $filters;

	/**
	 * @var int|bool
	 */
	#[ORM\Column(type: Types::SMALLINT, options: ['default' => 1])]
	public $filtersFromParent;

	#[Gedmo\Timestampable(on: 'create')]
	#[ORM\Column(type: Types::DATETIME_MUTABLE, options: ['default' => 'CURRENT_TIMESTAMP'])]
	protected DateTimeInterface $created;

	#[Gedmo\Timestampable(on: 'update')]
	#[ORM\Column(type: Types::DATETIME_MUTABLE, nullable: true)]
	protected ?DateTimeInterface $modified = null;

	/**
	 * Vlastní hodnoty třeba z rozšířeních
	 * @var ArrayHash|array|null
	 */
	#[ORM\Column(type: Types::ARRAY, nullable: true)]
	protected $attrs;

	#[ORM\Column(type: Types::STRING, length: 1, nullable: true)]
	public ?string $rod = null;

	/**
	 * @var int|bool
	 */
	#[ORM\Column(type: Types::SMALLINT, length: 1, options: ['default' => 1])]
	public $canProductsAddToCart;

	/**
	 * @var Collection<CategoryRelated>
	 */
	#[ORM\OneToMany(mappedBy: 'category', targetEntity: CategoryRelated::class)]
	protected Collection $related;

	public function __construct()
	{
		$this->categoryTexts        = new ArrayCollection;
		$this->categoryProducts     = new ArrayCollection;
		$this->children             = new ArrayCollection;
		$this->filters              = new ArrayCollection;
		$this->isPublished          = 0;
		$this->created              = new DateTime;
		$this->filtersFromParent    = 1;
		$this->attrs                = [];
		$this->canProductsAddToCart = 1;
		$this->extraFields          = new ArrayCollection;
		$this->related              = new ArrayCollection;
	}

	public function getLvl(): int { return $this->lvl; }

	public function getRoot(): ?Category { return $this->root; }

	public function addChildren(Category $child): void
	{
		$this->children->add($child);
	}

	public function addCategoryText(string $lang): void
	{
		$this->categoryTexts[$lang] = new CategoryTexts($this, $lang);
	}

	public function getCategoryText(?string $lang = null): ?CategoryTexts
	{
		return $this->categoryTexts[$lang ?? $this->locale];
	}

	/**
	 * @return Collection<string, CategoryTexts>
	 */
	public function getCategoryTexts(): Collection { return $this->categoryTexts; }

	/**
	 * @param CategoryTexts[] $categoryTexts
	 */
	public function setCategoryTexts(array $categoryTexts): self
	{
		$this->categoryTexts = new ArrayCollection($categoryTexts);

		return $this;
	}

	public function setCategoryText(CategoryTexts $categoryTexts): void
	{
		$this->categoryTexts->set($categoryTexts->getLang(), $categoryTexts);
	}

	public function getParent(): ?Category
	{
		return $this->parent;
	}

	public function setParent(?Category $parent): void
	{
		$this->parent = $parent;
	}

	public function addProduct(CategoryProduct $categoryProduct): void
	{
		$this->categoryProducts->add($categoryProduct);
	}

	public function getCreated(): DateTimeInterface { return $this->created; }

	public function getModified(): ?DateTimeInterface { return $this->modified ?: $this->getCreated(); }

	public function getAttrs(): array { return $this->attrs ? (array) $this->attrs : []; }

	/**
	 * @return mixed|null
	 */
	public function getAttr(string $key) { return $this->attrs[$key] ?? null; }

	/**
	 * @param mixed $value
	 */
	public function setAttr(string $key, $value): self
	{
		$this->attrs[$key] = $value;

		return $this;
	}

	public function removeAttr(string $key): self
	{
		unset($this->attrs[$key]);

		return $this;
	}

	public function getLft(): int { return $this->lft; }

	public function getRgt(): int { return $this->gt; }

}
