<?php declare(strict_types = 1);

namespace EshopCatalog\Model\Entities;

use Core\Model\Module;
use Doctrine;
use Doctrine\ORM\Event\PostRemoveEventArgs;
use Doctrine\ORM\Event\PostUpdateEventArgs;
use Doctrine\ORM\Event\PreFlushEventArgs;
use Doctrine\ORM\Event\PreRemoveEventArgs;
use Doctrine\ORM\Event\PreUpdateEventArgs;
use Doctrine\ORM\Mapping as ORM;
use Doctrine\ORM\Mapping\ClassMetadataInfo;
use EshopCatalog\FrontModule\Model\CacheService;
use EshopCatalog\Model\AvailabilityService;
use EshopCatalog\Model\Config;
use EshopCatalog\Model\Navigation\Home;
use EshopCatalog\Model\Products;
use Exception;
use Nette\Caching\Cache;
use Nette\SmartObject;
use Nette\Utils\DateTime;
use Nette\Utils\FileSystem;
use Nettrine\ORM\EntityManagerDecorator;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Tracy\Debugger;

class ProductListener implements EventSubscriberInterface
{
	use SmartObject;

	protected static array $cleared       = [];
	public static bool     $updateVariant = true;

	public function __construct(
		protected CacheService           $cacheService,
		protected AvailabilityService    $availabilityService,
		protected Products               $productsService,
		protected EntityManagerDecorator $em,
	)
	{
	}

	public static function getSubscribedEvents(): array
	{
		return [];
	}

	#[ORM\PreUpdate]
	public function preUpdate(Product $product, PreUpdateEventArgs $args): void
	{
		if (!$product->unlimitedQuantity) {
			// Aktualizuje dostupnost pokud se změnil počet skladem
			if ($product->getAvailability() && isset($args->getEntityChangeSet()['quantity'])) {
				$oldAvIdent = $product->getAvailability()->getIdent();
				$this->availabilityService->updateAvailabilityByQuantity($product);
				$newAvIdent = $product->getAvailability()->getIdent();

				if ($oldAvIdent != $newAvIdent) {
					/** @var ClassMetadataInfo $metadata */
					$metadata = $this->em->getClassMetadata(Product::class);
					$this->em->getConnection()->executeQuery(
						"UPDATE " . $metadata->getTableName() . " SET id_availability = ? WHERE id = ?",
						[$product->getAvailability()->getId(), $product->getId()]
					);
				}
			}
		}
	}

	#[ORM\PreFlush]
	public function onPreFlush(Product $product, PreFlushEventArgs $args): void
	{
		if (isset(self::$cleared[$product->getId()]) && self::$cleared[$product->getId()] === false) {
			$this->cacheService->clean('product', [
				Cache::Tags => [
					'product/' . $product->getId(),
				],
			]);

			$this->cacheService->clean('navigation', [
				Cache::Tags => [
					Home::CACHE_PRODUCT . '/' . $product->getId(),
				],
			]);
		}
		self::$cleared[$product->getId()] = true;
	}

	#[ORM\PostUpdate]
	public function postHandlerUpdate(Product $product, PostUpdateEventArgs $event): void
	{
		if (Module::isAdmin() && $product->isVariant() && static::$updateVariant) {
			$this->productsService->prepareUpdateVariant($product);
		}
	}

	#[ORM\PostUpdate]
	#[ORM\PostRemove]
	public function postHandler(Product $product, PostUpdateEventArgs|PostRemoveEventArgs $event): void
	{
		$em = $event->getObjectManager();

		if (Config::load('enableLoggingQuantityChange' . false)) {
			$changeSet = $em->getUnitOfWork()->getEntityChangeSet($product);
			if (isset($changeSet['quantity'])) {
				$old = $changeSet['quantity'][0];
				$new = $changeSet['quantity'][1];

				try {
					$dir  = '_quantities/' . (new DateTime())->format('Y-m-d');
					$file = $product->getId();
					FileSystem::createDir(LOG_DIR . '/' . $dir);
					$e = new Exception();
					Debugger::log(print_r($old . ' -> ' . $new, true), $dir . '/' . $file);
					Debugger::log('New availability: ' . $product->getAvailability()->getIdent(), $dir . '/' . $file);

					if (isset($changeSet['availability'])) {
						if ($changeSet['availability']) {
							Debugger::log(
								'Can add to cart: ' . print_r(
									$changeSet['availability'][0]->canAddToCart . ' -> ' . $changeSet['availability'][1]->canAddToCart,
									true,
								),
								$dir . '/' . $file,
							);
						}

						if ($changeSet['availability']) {
							Debugger::log(
								'Open graph text: ' . print_r(
									$changeSet['availability'][0]->openGraphText . ' -> ' . $changeSet['availability'][1]->openGraphText,
									true,
								),
								$dir . '/' . $file,
							);
						}
					}
				} catch (Exception $e) {
				}
			}
		}

		self::$cleared[$product->getId()] = true;
	}

	#[ORM\PreRemove]
	public function preRemoveHandler(Product $product, PreRemoveEventArgs $events): void
	{
		$this->cacheService->productCache->clean([Cache::Tags => ['product/' . $product->getId()]]);

		if ($product->getGallery() && !$product->isVariant()) {
			$events->getObjectManager()->remove($product->getGallery());
			$events->getObjectManager()->flush();
		}
	}

}
