<?php declare(strict_types = 1);

namespace EshopCatalog\Model\Entities;

use Core\Model\Entities\Country;
use Core\Model\Entities\TId;
use Core\Model\Helpers\Strings;
use DateTimeInterface;
use Doctrine\DBAL\Types\Types;
use Doctrine\ORM\Mapping as ORM;
use EshopCatalog\Model\Listeners\ProductPriceCountryHistoryListener;
use Nette\Utils\DateTime;

#[ORM\Table(name: 'eshop_catalog__product_price_history_country')]
#[ORM\Index(columns: ['direction', 'country', 'currency_code', 'product'], name: 'byCountry')]
#[ORM\Entity]
#[ORM\EntityListeners([ProductPriceCountryHistoryListener::class])]
class ProductPriceCountryHistory
{
	use TId;

	#[ORM\ManyToOne(targetEntity: Product::class)]
	#[ORM\JoinColumn(name: 'product', referencedColumnName: 'id', onDelete: 'CASCADE')]
	public Product $product;

	#[ORM\ManyToOne(targetEntity: Country::class)]
	#[ORM\JoinColumn(name: 'country', referencedColumnName: 'id', nullable: false, onDelete: 'CASCADE')]
	public Country $country;

	#[ORM\Column(name: 'currency_code', type: Types::STRING, length: 255, nullable: true)]
	public ?string $currency = null;

	/**
	 * @var double|string
	 */
	#[ORM\Column(type: Types::DECIMAL, precision: 10, scale: 2, nullable: false)]
	public $price;

	#[ORM\Column(type: Types::DATETIME_MUTABLE, nullable: false, options: ['default' => 'CURRENT_TIMESTAMP'])]
	protected DateTimeInterface $created;

	#[ORM\Column(type: Types::SMALLINT, nullable: false, options: ['default' => 0])]
	public int $direction = 0;

	#[ORM\Column(type: Types::STRING, length: 255, nullable: true)]
	public ?string $reason = null;

	public function __construct(
		Product $product,
		Country $country,
		?float  $price,
		float   $newPrice,
		?string $currency = null,
		?string $reason = null,
	)
	{
		if (!$price) {
			$price = 0.0;
		}

		$this->product   = $product;
		$this->country   = $country;
		$this->price     = Strings::formatEntityDecimal($price);
		$this->created   = new DateTime;
		$this->direction = $newPrice > $price ? 1 : 0;
		$this->currency  = $currency;
		$this->reason    = $reason;
	}

	public function getCreated(): DateTimeInterface { return $this->created; }

}
