<?php declare(strict_types = 1);

namespace EshopCatalog\Model\Entities;

use Core\Model\Entities\TId;
use Core\Model\Helpers\Strings;
use DateTimeInterface;
use Doctrine\DBAL\Types\Types;
use Doctrine\ORM\Mapping as ORM;
use EshopCatalog\Model\Listeners\ProductPriceHistoryListener;
use Nette\Utils\DateTime;

#[ORM\Table(name: 'eshop_catalog__product_price_history')]
#[ORM\Index(columns: ['direction', 'product'], name: 'byCountry')]
#[ORM\Entity]
#[ORM\EntityListeners([ProductPriceHistoryListener::class])]
class ProductPriceHistory
{
	use TId;

	#[ORM\ManyToOne(targetEntity: Product::class)]
	#[ORM\JoinColumn(name: 'product', referencedColumnName: 'id', onDelete: 'CASCADE')]
	public Product $product;

	/**
	 * @var double|string
	 */
	#[ORM\Column(type: Types::DECIMAL, precision: 10, scale: 2, nullable: false)]
	public $price;

	#[ORM\Column(type: Types::DATETIME_MUTABLE, nullable: false, options: ['default' => 'CURRENT_TIMESTAMP'])]
	protected DateTimeInterface $created;

	#[ORM\Column(type: Types::SMALLINT, nullable: false, options: ['default' => 0])]
	public int $direction = 0;

	#[ORM\Column(type: Types::STRING, length: 255, nullable: true)]
	public ?string $reason = null;

	public function __construct(
		Product $product,
		?float  $price,
		float   $newPrice,
		?string $reason = null,
	)
	{
		if (!$price) {
			$price = 0.0;
		}

		$this->product   = $product;
		$this->price     = Strings::formatEntityDecimal($price);
		$this->created   = new DateTime;
		$this->direction = $newPrice > $price ? 1 : 0;
		$this->reason    = $reason;
	}

	public function getCreated(): DateTimeInterface { return $this->created; }

}
