<?php declare(strict_types = 1);

namespace EshopCatalog\Model\Navigation;

use Core\Components\Navigation\DaoNavigationItem;
use Core\Model\Dao\SiteMapUrl;
use Core\Model\Sites;
use Core\Model\UI\Form\BaseContainer;
use Core\Model\UI\Form\BaseForm;
use Doctrine\ORM\NonUniqueResultException;
use EshopCatalog\FrontModule\Model\Categories;
use EshopCatalog\FrontModule\Model\ProductQuery;
use EshopCatalog\FrontModule\Model\ProductsFacade;
use EshopCatalog\Model\Config;
use EshopCatalog\Model\Event\RouteInFindAliasEvent;
use EshopCatalog\Model\Helpers\ProductsHelper;
use Navigations\AdminModule\Components\NavigationForm;
use Navigations\Model\Event\RouteInEvent;
use Navigations\Model\Event\RouteOutEvent;
use Navigations\Model\Helper\NavigationsHelper;
use Navigations\Model\NavigationConfig;
use Navigations\Model\Providers\INavigationItem;
use Nette\Application\LinkGenerator;
use Nette\Application\UI\InvalidLinkException;
use Nette\Caching\Cache;
use Nette\DI\Container;
use Nette\Utils\ArrayHash;
use Nette\Utils\DateTime;
use Nette\Utils\Strings;
use Pages\AdminModule\Model\TemplatePages;
use Throwable;
use function str_ends_with;

class Home extends BaseNavigation implements INavigationItem
{
	final public const COMPONENT_NAME    = 'eshopCatalog.navigation.home';
	final public const DEFAULT_PRESENTER = 'EshopCatalog:Front:Default';
	final public const ACTION_PRODUCT    = 'product';
	final public const ACTION_CATEGORY   = 'category';
	final public const CACHE_PRODUCT     = 'eshopCatalogProduct';

	protected $title  = 'home';
	protected $action = 'default';

	public ?LinkGenerator $linkGenerator       = null;
	protected int         $componentChildStart = 0;
	protected int         $componentChildLimit = 4000;
	protected ?string     $lastSite            = null;

	public function __construct(
		protected TemplatePages  $templatePagesService,
		protected ProductsFacade $productsFacade,
		protected ProductsHelper $productsHelper,
		protected Sites          $sites,
		protected Container      $container,
		protected Categories     $categories,
	)
	{
	}

	/**
	 * @inheritDoc
	 */
	public function getFormContainer(BaseForm $baseForm)
	{
		$container = new BaseContainer;

		$site  = $this->navFormSite ?? $baseForm->getCustomData('loadComponent')['site'];
		$pages = $this->templatePagesService->getOptionsForSelectGrouped();
		$pages = (array) ($pages[$site ?? $baseForm->getComponent('site')->getValue()] ?? $pages);

		$container->addSelect('templateHome', 'eshopCatalog.navigationBuilder.templateHome', $pages);
		$container->addSelect('templateProduct', 'eshopCatalog.navigationBuilder.templateDetail', $pages);
		$container->addSelect('templateCategory', 'eshopCatalog.navigationBuilder.templateCategory', $pages);

		/** @var NavigationForm $parent */
		$parent = $baseForm->getParent();
		if ($parent->lookup(null, false)) {
			$componentId = $parent->getParameter('componentId');
			if ($componentId) {
				$container->getComponent('templateHome')
					->setHtmlAttribute('data-custom-link', $parent->link('componentCustom!', [$componentId]));
			}
		} else {
			$baseForm->onAnchor[] = static function() use ($container, $baseForm, $parent) {
				$container->getComponent('templateHome')
					->setHtmlAttribute(
						'data-custom-link',
						$parent->link('componentCustom!', [$baseForm->getValues()->componentType]),
					);
			};
		}

		return $container;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values)
	{

	}

	public function updateCacheDep(array &$cache, $urlParams)
	{
		$cache[Cache::Tags][] = self::CACHE_PRODUCT . '/' . $urlParams['id'];

		parent::updateCacheDep($cache, $urlParams);
	}

	/*******************************************************************************************************************
	 * ===========================  Route
	 */

	public function presenterActionCheck($presenter, $action)
	{
		return self::DEFAULT_PRESENTER == $presenter && in_array($action, [$this->action, self::ACTION_PRODUCT,
				self::ACTION_CATEGORY]);
	}

	/**
	 * @param DaoNavigationItem $navigation
	 * @param array             $urlParams
	 *
	 * @return array|mixed
	 * @throws NonUniqueResultException
	 * @throws InvalidLinkException
	 * @throws Throwable
	 */
	public function routerIn($navigation, $urlParams = [])
	{
		$returnDefault = [
			'presenter' => self::DEFAULT_PRESENTER,
			'action'    => $this->action,
			'id'        => null,
		];

		if (isset($urlParams['getHomepage'])) {
			return $returnDefault;
		}

		if (isset($urlParams['path']) && $navigation->alias != $urlParams['path']) {
			$params    = explode('/', (string) $urlParams['path']);
			$lastParam = NavigationsHelper::removeUrlSuffix(end($params));
			$id        = explode('-', $lastParam, 2);
			$resolver  = Config::load('useOldUrls') ? $this->productsHelper->getOldUrlResolver() : null;

			if ($resolver) {
				$return = $resolver->routeIn($navigation, $urlParams, $returnDefault);

				if ($return) {
					return $return;
				}
			}

			if (Config::load('product.removeIdFromUrl')) {
				$pathWithoutProduct = Strings::substring($urlParams['path'], 0, -Strings::length(end($params)));
				$pathWithoutProduct = Strings::trim($pathWithoutProduct, '/');

				$aliasId = null;
				foreach ($this->productsFacade->productsService->findIdByAlias($lastParam) as $row) {
					$categoryPrefix = $this->getCategoryPath((int) $row['category']);

					if (str_ends_with($pathWithoutProduct, trim($categoryPrefix, '/'))) {
						$aliasId = $row['id'];
						break;
					}

					if ($aliasId === null) {
						$aliasId = $row['id'];
					}
				}

				if ($aliasId) {
					return [
						'presenter' => self::DEFAULT_PRESENTER,
						'action'    => self::ACTION_PRODUCT,
						'id'        => $aliasId,
					];
				}
			}

			if (is_numeric($id[0]) && $this->productsFacade->getProduct((int) $id[0])) {
				return [
					'presenter' => self::DEFAULT_PRESENTER,
					'action'    => self::ACTION_PRODUCT,
					'id'        => $id[0],
				];
			}

			$return = [];
			$this->eventDispatcher->dispatch(
				new RouteInFindAliasEvent($urlParams['path'], $urlParams, $return),
				BaseNavigation::class . '::routeInFindAlias',
			);
			if ($return) {
				if (!isset($return['activeNavigation'])) {
					$return['activeNavigation'] = $navigation;
				}

				return $return;
			}
		}

		$returnEvent = [];
		$this->eventDispatcher->dispatch(
			new RouteInEvent(null, $urlParams, $returnEvent),
			Home::class . '::routeInStart',
		);
		if ($returnEvent) {
			return $returnEvent;
		}

		if ($urlParams['path'] === $navigation->alias) {
			return $returnDefault;
		}

		return null;
	}

	public function routerOut(DaoNavigationItem $navigation, &$urlParams): ?string
	{
		// Modifikace url pri pouziti virtualni kategorie
		if ($navigation->getParam('url')) {
			$cfUrl = $navigation->getParam('url');
		} else {
			$cfUrl = $navigation->isHomepage ? '/' : '/' . $navigation->alias;
		}

		if ($urlParams['action'] == self::ACTION_PRODUCT && isset($urlParams['id'])) {
			$resolver = Config::load('useOldUrls') ? $this->productsHelper->getOldUrlResolver() : null;

			if ($resolver) {
				$out = $resolver->routeOut($navigation, $urlParams);

				if ($out) {
					return $out;
				}
			}

			$useBaseCategoryInUrl = (bool) Config::load('product.useBaseCategoryInUrl');

			$product = $this->productsFacade->productsService->getForNavigation(
				(int) $urlParams['id'],
				$useBaseCategoryInUrl,
				$urlParams['locale'],
			);

			if ($product) {
				$return = ($navigation->isHomepage ? '' : '/' . $navigation->alias);

				if (!Config::load('product.removeIdFromUrl')) {
					$return .= '/' . $product['id'] . '-';
				}

				$return .= Strings::webalize($product['alias']);

				if ($useBaseCategoryInUrl && $product['category']) {
					$categoryPrefix = $this->getCategoryPath((int) $product['category']);

					if ($categoryPrefix) {
						$return = '/' . $categoryPrefix . '/' . trim($return, '/');
					}
				}

				if (NavigationConfig::load('urlSuffix')) {
					$return .= NavigationConfig::load('urlSuffix');
				}

				return $return;
			}
		} else {
			$this->eventDispatcher->dispatch(
				new RouteOutEvent($urlParams, $navigation, $cfUrl),
				BaseNavigation::class . '::routeOut',
			);
		}

		return '/' . ltrim((string) $cfUrl, '/');
	}

	/**
	 * @inheritDoc
	 */
	public function getComponentChild(string $lang, $navigation = null, $siteMapData = false)
	{
		$arr         = [];
		$currentSite = $this->sites->getCurrentSite()->getIdent();

		if ($this->lastSite != $currentSite) {
			$this->componentChildStart = 0;
		}

		foreach ((new ProductQuery($lang))
			         ->withTexts()
			         ->inSite($currentSite)
			         ->getQueryBuilder($this->productsFacade->productsService->getEr())
			         ->select('p.id, pt.name, p.modified, pt.seo, p.modified')
			         ->andWhere('sites.isActive = 1')
			         ->andWhere('p.isDeleted = 0')
			         ->andWhere('p.disableListing = 0')
			         ->setFirstResult($this->componentChildStart)->setMaxResults($this->componentChildLimit)
			         ->getQuery()->getScalarResult() as $row) {
			$id = $row['id'];

			$data = [
				'presenter' => self::DEFAULT_PRESENTER,
				'action'    => self::ACTION_PRODUCT,
				'id'        => $id,
				'locale'    => $lang,
			];

			if ($siteMapData) {
				$siteMapUrl = new SiteMapUrl();
				if ($row['modified']) {
					$siteMapUrl->setModified(DateTime::from($row['modified']));
				}

				$seo = $row['seo'] ? @unserialize($row['seo']) : null;

				if (!$seo) {
					$seo = [];
					foreach (['addToSiteMap', 'siteMapChangeFreq', 'siteMapPriority', 'robots'] as $col) {
						$seo[$col] = $navigation->getSeo($col);
					}
				}

				if (is_array($seo)) {
					if (!isset($seo['addToSiteMap'])) {
						$seo['addToSiteMap'] = '1';
					}
					if ($seo['addToSiteMap'] === '0' || is_string($seo['robots']) && str_contains(
							'noindex',
							$seo['robots'],
						)) {
						continue;
					}

					$siteMapUrl->setSeo($seo);
				}

				$data['siteMapData']['default'] = $siteMapUrl;
			}

			$arr[] = $data;
		}

		$this->productsFacade->productsService->em->clear();

		$this->componentChildStart += $this->componentChildLimit;
		$this->lastSite            = $currentSite;

		if (!empty($arr)) {
			$arr['continue'] = true;
		}

		return $arr;
	}

	protected function getCategoryPath(int $catId): string
	{
		$path = $this->categories->getQueryPath($catId);
		array_shift($path);

		return implode('/', array_map(static fn($v) => $v['alias'], $path));
	}

}
