<?php declare(strict_types = 1);

namespace EshopCatalog\Model\Subscribers;

use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Entities\ExtraField;
use EshopCatalog\FrontModule\Model\Categories;
use EshopCatalog\Model\Config;
use EshopCatalog\Model\Entities\Category;
use EshopCatalog\Model\Event\RouteInFindAliasEvent;
use EshopCatalog\Model\Navigation\BaseNavigation;
use EshopCatalog\Model\Navigation\Home;
use EshopCatalog\Model\NavigationFilter;
use Navigations\Model\Event\RouteInEvent;
use Navigations\Model\Event\RouteOutEvent;
use Navigations\Model\Event\RouteUpdateCacheDep;
use Navigations\Model\Navigations;
use Nette\Caching\Cache;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

class NavigationSubscriber implements EventSubscriberInterface
{
	protected static array $outFilter        = [];
	protected array        $cRouteInNotFound = [];

	public function __construct(
		protected EntityManagerDecorator $entityManager,
		protected NavigationFilter       $navigationFilter,
		protected Categories             $categories,
		protected Navigations            $navigations,
	)
	{
	}

	public static function getSubscribedEvents(): array
	{
		return [
			BaseNavigation::class . '::routeOut'            => ['routeOut', 150],
			BaseNavigation::class . '::routeUpdateCacheDep' => ['routeUpdateCacheDep', 150],
			Navigations::class . '::routeInNotFound'        => ['routeInNotFound', 150],
			Home::class . '::routeInStart'                  => ['routeInNotFound', 150],
			BaseNavigation::class . '::routeInFindAlias'    => ['routeInNotFound', 150],
		];
	}

	public function routeUpdateCacheDep(RouteUpdateCacheDep $event): void
	{
		if (!empty(self::$outFilter)) {
			foreach (self::$outFilter as $type => $v) {
				foreach ($v as $k => $vv) {
					$event->cache[Cache::Tags][] = 'categoriesFilter_' . $type . '_' . $k;
				}
			}

			self::$outFilter = [];
		}
	}

	public function routeOut(RouteOutEvent $event): void
	{
		if ($event->disableDefaultFilter === true) {
			return;
		}

		$urlParams        = &$event->urlParams;
		$event->urlParams = $this->navigationFilter->processRouterOut($urlParams);

		if ($urlParams['productsFilter-filter']) {
			self::$outFilter = $urlParams['productsFilter-filter'];
		}
	}

	public function routeInNotFound(
		RouteInEvent|RouteInFindAliasEvent $event,
	): void
	{
		if (!Config::load('allowCategoryFullUrlField', false)) {
			return;
		}

		$key = $event->urlParams['path'] . '_' . $event->urlParams['locale'];
		if (!isset($this->cRouteInNotFound[$key])) {
			$this->cRouteInNotFound[$key] = [];
			$resultCache                  = Config::load('productsList.enableResultCacheOther');

			$query = $this->entityManager->getRepository(ExtraField::class)->createQueryBuilder('ef')
				->select('ef.sectionKey')
				->where('ef.sectionName = :sn')
				->andWhere('ef.key = :key')
				->andWhere('ef.value IN (:value)')
				->andWhere('ef.lang = :lang')
				->setParameters([
					'sn'    => Category::EXTRA_FIELD_SECTION,
					'key'   => 'fullUrl',
					'value' => [
						ltrim((string) $event->urlParams['path'], '/'),
						'/' . ltrim((string) $event->urlParams['path'], '/'),
					],
					'lang'  => $event->urlParams['locale'],
				])->getQuery();

			if ($resultCache) {
				$query->enableResultCache($resultCache);
			}

			$catIds = $query->getScalarResult();

			foreach ($catIds as $catId) {
				$catId = $catId['sectionKey'];

				if ($catId) {
					$catId = (int) $catId;
					$cat   = $this->categories->get($catId);

					if ($cat) {
						$r   = $cat->getParent();
						$nav = null;
						$i   = 0;
						while ($r && $i <= 10) {
							$nav = $this->categories->findNavigationId($r->getId());

							if ($nav) {
								break;
							}

							$r = $r->getParent();
							$i++;
						}

						if ($nav) {
							$this->cRouteInNotFound[$key] = [
								'nav'              => $this->navigations->getNavigation($nav),
								'presenter'        => 'EshopCatalog:Front:Default',
								'action'           => 'category',
								'id'               => $catId,
								'activeNavigation' => $nav,
							];
						}
					}
				}
			}
		}

		if (!empty($this->cRouteInNotFound[$key])) {
			$event->return['presenter']        = $this->cRouteInNotFound[$key]['presenter'];
			$event->return['action']           = $this->cRouteInNotFound[$key]['action'];
			$event->return['id']               = $this->cRouteInNotFound[$key]['id'];
			$event->return['activeNavigation'] = $this->cRouteInNotFound[$key]['nav'];
			$event->stopPropagation();
		}
	}

}
