<?php declare(strict_types = 1);

namespace EshopCatalog\AdminModule\Components\Features;

use Core\Components\Flashes\Flashes;
use Core\Model\UI\BaseControl;
use Core\Model\UI\DataGrid\BaseDataGrid;
use Core\Model\UI\DataGrid\DataSource\DoctrineDataSource;
use Doctrine\ORM\Query\Expr\Join;
use EshopCatalog\AdminModule\Model\FeatureValues;
use EshopCatalog\Model\Entities\FeatureProduct;
use Nette\Application\AbortException;
use Nette\Utils\Html;
use EshopCatalog\Model\Entities\Feature;
use EshopCatalog\AdminModule\Model\Features;

class FeaturesGrid extends BaseControl
{
	protected Features      $featureServices;
	protected FeatureValues $featureValuesService;

	public function __construct(
		Features      $features,
		FeatureValues $featureValues
	)
	{
		$this->featureServices      = $features;
		$this->featureValuesService = $featureValues;
	}

	public function render(): void
	{
		$this->template->setFile(__DIR__ . '/FeaturesGrid.latte');
		$this->template->render();
	}

	/*******************************************************************************************************************
	 * ==================  Handle
	 */

	public function handleMergeAll(): void
	{
		$this->featureServices->mergeAll();
		$this->redirect('this');
	}

	public function handleMergeAllValues(): void
	{
		$this->featureValuesService->mergeAll();
		$this->redirect('this');
	}

	public function handleRepairPositions(): void
	{
		$this->featureServices->repairPositions();
		$this->redirect('this');
	}

	public function handleRepairValuesPositions(): void
	{
		$this->featureValuesService->repairPositions();
		$this->redirect('this');
	}

	public function handleSortAlphabeticallyFeatures(): void
	{
		$this->featureServices->sortAlphabetically();
		$this->redirect('this');
	}

	public function handleSortAlphabeticallyFeatureValues(): void
	{
		$this->featureValuesService->sortAlphabetically();
		$this->redirect('this');
	}

	/*******************************************************************************************************************
	 * ==================  Components
	 */

	protected function createComponentGrid(): BaseDataGrid
	{
		$grid = $this->createGrid();
		$grid->setDefaultPerPage(50);
		$grid->setRememberState();

		$qb = $this->featureServices->getEr()->createQueryBuilder('f')->addSelect('ft')
			->join('f.featureTexts', 'ft', 'WITH', 'ft.lang = :lang')->setParameter('lang', $this->translator->getLocale())
			->orderBy('f.position');
		$grid->setDataSource($qb);

		$productsWithFilter = [];

		/** @var DoctrineDataSource $ds */
		$ds                 = $grid->getDataSource();
		$ds->onDataLoaded[] = function($items) use (&$productsWithFilter) {
			$ids = array_map(static function($v) { return $v->getId(); }, $items);

			foreach ($this->em->getRepository(FeatureProduct::class)->createQueryBuilder('fp')
				         ->select('count(IDENTITY(fp.product)) as count, IDENTITY(fv.feature) as feature')
				         ->innerJoin('fp.featureValue', 'fv', Join::WITH, 'fv.feature IN (:ids)')
				         ->setParameter('ids', $ids)
				         ->groupBy('fv.feature')->getQuery()->getScalarResult() as $row) {
				$productsWithFilter[$row['feature']] = (int) $row['count'];
			}
		};

		$grid->setSortable();
		$grid->setSortableHandler('featuresGrid:gridSortableRow!');

		// Columns
		$grid->addColumnText('name', 'eshopCatalog.defaultGrid.name')->setRenderer(function($row) {
			return Html::el('a', ['href' => $this->getPresenter()->link('VariantsFeatures:editFeature', $row->getId())])
				->setText($row->getFeatureText($this->translator->getLocale())->name);
		});
		$grid->addColumnStatus('useAsFilter', 'eshopCatalog.defaultGrid.useAsFilter')->setAlign('center')
			->addOption(1, 'eshopCatalog.defaultGrid.enabled')->setIcon('check')->setClass('btn-success')->setShowTitle(false)->endOption()
			->addOption(0, 'eshopCatalog.defaultGrid.unabled')->setIcon('times')->setClass('btn-danger')->setShowTitle(false)->endOption()
			->onChange[] = [$this, 'gridEnabledFilterChange'];
		$grid->addColumnStatus('isPublished', 'default.isPublished')->setAlign('center')
			->addOption(1, 'default.publish')->setIcon('check')->setClass('btn-success')->setShowTitle(false)->endOption()
			->addOption(0, 'default.unPublish')->setIcon('times')->setClass('btn-danger')->setShowTitle(false)->endOption()
			->onChange[] = [$this, 'gridPublishChange'];

		// Filter
		$grid->addFilterText('name', '', 'ft.name');
		$grid->addFilterSelect('useAsFilter', '', [
			''  => '',
			'1' => $this->t('default.yes'),
			'0' => $this->t('default.no'),
		]);
		$grid->addFilterSelect('isPublished', '', [
			''  => '',
			'1' => $this->t('default.yes'),
			'0' => $this->t('default.no'),
		]);

		// Actions
		$grid->addAction('edit', '', 'VariantsFeatures:editFeature')->setIcon('edit')->setBsType('primary');
		$grid->addAction('delete', '', 'delete!')
			->setIcon('times')->setBsType('danger')->addClass('ajax')
			->setConfirm(function(Feature $feature) use (&$productsWithFilter) {
				$c = $productsWithFilter[$feature->getId()];

				return $this->t('eshopCatalog.featuresGrid.' . ($c ? 'reallyDeleteFeatureWithProducts' : 'reallyDeleteFeature'), [
					'feature' => $feature->getFeatureText()->name,
					'prod'    => $c ? number_format($c, 0, '', ' ') : '',
				]);
			});

		// Prototype
		$grid->getColumn('useAsFilter')->getElementPrototype('th')->class[] = 'w1nw';
		$grid->getColumn('isPublished')->getElementPrototype('td')->class[] = 'w1nw';

		return $grid;
	}

	/*******************************************************************************************************************
	 * ==================  Handle
	 */

	/**
	 * @param int|string $id
	 *
	 * @throws AbortException
	 */
	public function handleDelete($id): void
	{
		$presenter = $this->getPresenter();
		if ($this->featureServices->remove($id)) {
			$presenter->flashMessage('eshopCatalog.defaultGrid.removed', Flashes::FLASH_SUCCESS);
		} else {
			$presenter->flashMessage('eshopCatalog.defaultGrid.removeFailed', Flashes::FLASH_DANGER);
		}

		if ($presenter->isAjax()) {
			$this['grid']->reload();
			$presenter->redrawControl('flashes');
		} else {
			$presenter->redirect('this');
		}
	}

	/*******************************************************************************************************************
	 * =================  Grid function
	 */

	/**
	 * @param int      $id
	 * @param int|bool $newStatus
	 *
	 * @throws AbortException
	 */
	public function gridPublishChange($id, $newStatus): void
	{
		$presenter = $this->getPresenter();

		if ($this->featureServices->setPublish($id, $newStatus)) {
			$presenter->flashMessageSuccess('eshopCatalog.defaultGrid.publishChanged');
		} else {
			$presenter->flashMessageDanger('eshopCatalog.defaultGrid.publishChangeFailed');
		}

		if ($presenter->isAjax()) {
			$this['grid']->redrawItem($id);
			$presenter->redrawControl('flashes');
		} else {
			$presenter->redirect('this');
		}
	}

	/**
	 * @param int      $id
	 * @param int|bool $newStatus
	 *
	 * @throws AbortException
	 */
	public function gridEnabledFilterChange($id, $newStatus): void
	{
		$presenter = $this->getPresenter();

		if ($this->featureServices->setUseAsFilter($id, $newStatus)) {
			$presenter->flashMessageSuccess('eshopCatalog.defaultGrid.filterChanged');
		} else {
			$presenter->flashMessageDanger('eshopCatalog.defaultGrid.filterChangeFailed');
		}

		if ($presenter->isAjax()) {
			$this['grid']->redrawItem($id);
			$presenter->redrawControl('flashes');
		} else {
			$presenter->redirect('this');
		}
	}

	public function handleGridSortableRow(): void
	{
		$presenter = $this->getPresenter();
		$request   = $presenter->getHttpRequest();
		$id        = $request->getPost('id');
		$position  = $request->getPost('position');

		if ($id != null && $position != null && $this->featureServices->setPosition($id, $position)) {
			$presenter->flashMessageSuccess('default.positionChanged');
		} else {
			$presenter->flashMessageDanger('default.positionChangeFailed');
		}

		if ($presenter->isAjax()) {
			$presenter->redrawControl('flashes');
			$this['grid']->reload();
		} else {
			$this->redirect('this');
		}
	}

}
