<?php declare(strict_types = 1);

namespace EshopCatalog\AdminModule\Components\Manufacturers;

use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use Exception;
use Nette\Application\BadRequestException;
use Nette\Utils\ArrayHash;
use EshopCatalog\Model\Entities\Manufacturer;
use EshopCatalog\Model\Entities\ManufacturerTexts;
use EshopCatalog\AdminModule\Model\Manufacturers;

class ManufacturerForm extends BaseControl
{
	public ?Manufacturer    $manufacturer = null;
	protected Manufacturers $manufacturerServices;

	public function __construct(
		Manufacturers $manufacturers
	)
	{
		$this->manufacturerServices = $manufacturers;
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	protected function createComponentForm(): BaseForm
	{
		$form = $this->createForm();

		$form->addText('name', 'eshopCatalog.manufacturerForm.name')
			->setRequired()
			->setMaxLength(255);
		$form->addBool('isPublished', 'eshopCatalog.manufacturerForm.isPublished')
			->setDefaultValue(0);
		$form->addText('website', 'eshopCatalog.manufacturerForm.website')
			->setNullable()
			->setMaxLength(255);
		$form->addFilesManager('logo', 'eshopCatalog.manufacturerForm.logo');
		$form->addEditor('desc', 'eshopCatalog.manufacturerForm.desc')
			->setHeight(300)
			->setIsMultilanguage();

		$form->addSaveCancelControl();

		$form->onSuccess[] = [$this, 'formSuccess'];

		return $form;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values): bool
	{
		try {
			$manufacturer = $this->manufacturer ?: new Manufacturer;

			$manufacturer->name        = $values->name;
			$manufacturer->isPublished = $values->isPublished;
			$manufacturer->website     = $values->website;
			$manufacturer->logo        = $values->logo;

			$mTexts = $form->prepareMultilangTextsForEntity(['desc' => 'description'], (array) $values);
			foreach ($mTexts as $lang => $vals) {
				$text = $manufacturer->getText($lang) ?: new ManufacturerTexts($manufacturer, $lang);
				foreach ($vals as $k => $v) {
					$text->$k = $v;
				}

				$this->em->persist($text);
			}

			$this->em->persist($manufacturer);
			$this->em->flush();
			$form->addCustomData('manufacturerId', $manufacturer->getId());
			$this->getPresenter()->flashMessageSuccess('eshopCatalog.manufacturerForm.manufacturerSaved');
		} catch (Exception $e) {
			$form->addError($e->getMessage());

			return false;
		}

		return true;
	}

	/**
	 * @param int|string $id
	 *
	 * @throws BadRequestException
	 */
	public function setManufacturer($id): void
	{
		$this->manufacturer = $this->em->getRepository(Manufacturer::class)->find($id);

		if ($this->manufacturer) {
			$form = $this['form'];
			$form->setDefaults([
				'name'        => $this->manufacturer->name,
				'isPublished' => $this->manufacturer->isPublished,
				'website'     => $this->manufacturer->website,
				'logo'        => $this->manufacturer->logo,
			]);

			$mTexts = $form->prepareMultilangTextsForForm(['desc' => 'description'], $this->manufacturer->getTexts());
			$form->setDefaults($mTexts['texts']);
		} else {
			$this->getPresenter()->error();
		}
	}

}
