<?php declare(strict_types = 1);

namespace EshopCatalog\AdminModule\Components\Products;

use Core\Model\UI\AbstractPresenter;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use EshopCatalog\AdminModule\Model\FormContainers\RelatedProductContainer;
use EshopCatalog\AdminModule\Model\Products;
use EshopCatalog\FrontModule\Model\CacheService;
use EshopCatalog\Model\Helpers\VariantsHelper;
use Nette\Application\IPresenter;
use Nette\Utils\ArrayHash;

class RelatedProductsForm extends BaseControl
{
	protected RelatedProductContainer $relatedProductContainer;
	protected Products                $products;
	protected VariantsHelper          $variantsHelper;
	protected CacheService            $cacheService;

	public function __construct(
		RelatedProductContainer $relatedProductContainer,
		Products                $products,
		VariantsHelper          $variantsHelper,
		CacheService            $cacheService
	)
	{
		$this->relatedProductContainer = $relatedProductContainer;
		$this->products                = $products;
		$this->variantsHelper          = $variantsHelper;
		$this->cacheService            = $cacheService;
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	public function createComponentForm(): BaseForm
	{
		$form = $this->createForm();
		$form->addComponent($this->relatedProductContainer->getContainer(), 'relatedProducts');

		$this->monitor(IPresenter::class, function(AbstractPresenter $presenter) use ($form) {
			$form->getComponent('relatedProducts')['product']->getControlPrototype()->setAttribute('data-link', $presenter->link(':EshopCatalog:Cron:Products:loadAll'));
		});
		$form->addCheckbox('applyToVariants', 'eshopCatalog.productFeature.applyToVariants');

		$form->addSubmit('submit', 'default.save');

		$form->onSuccess[] = [$this, 'formSuccess'];

		return $form;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values): void
	{
		/** @var int[] $productsIds */
		$productsIds = explode('-', (string) $this->presenter->getParameter('ids'));

		if ($values->applyToVariants ?? false) {
			$productsIds = array_merge($productsIds, ...$this->variantsHelper->getProductVariants($productsIds));
		}

		if (count($productsIds) > 0) {
			foreach ($productsIds as $productId) {
				if (!$product = $this->products->get((int) $productId)) {
					continue;
				}
				$this->relatedProductContainer->saveData($form->getHttpData(), $product);
				$this->em->persist($product);
			}

			$this->em->flush();

			foreach ($productsIds as $productsId) {
				$this->cacheService->productCache->remove('related/' . $productsId);
			}

			$this->getPresenter()->flashMessageSuccess('default.saved');
		}
	}

}
