<?php declare(strict_types = 1);

namespace EshopCatalog\AdminModule\Model\Helpers;

use Core\Model\Entities\Country;
use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Helpers\Strings;
use EshopCatalog\Model\Config;
use EshopCatalog\Model\Entities\Product;
use EshopCatalog\Model\Entities\ProductPrice;
use EshopCatalog\Model\Entities\ProductPriceCountryHistory;
use EshopCatalog\Model\Entities\ProductPriceLevelCountry;
use EshopCatalog\Model\Entities\VatRate;
use EshopOrders\Model\Entities\GroupCustomers;

class PricesHelper
{
	protected EntityManagerDecorator $em;

	public function __construct(EntityManagerDecorator $em)
	{
		$this->em = $em;
	}

	public function savePricesToProduct(Product $product, array $data): void
	{
		$currentPrices = $product->prices->toArray();

		// Create / Update
		foreach ($data as $countryId => $values) {
			if (!$values) {
				continue;
			}

			if (isset($currentPrices[Strings::lower($countryId)])) {
				$countryId = Strings::lower($countryId);
				$price     = $currentPrices[$countryId];
			} else if (isset($currentPrices[Strings::upper($countryId)])) {
				$countryId = Strings::upper($countryId);
				$price     = $currentPrices[$countryId];
			} else {
				$price = new ProductPrice($product, $this->em->getReference(Country::class, $countryId), $values['currency']);
			}

			if (Config::load('enablePriceHistory') && $price->price !== $values['price'] && $values['price']) {
				$priceHistory = new ProductPriceCountryHistory(
					$product,
					$this->em->getReference(Country::class, $countryId),
					(float) $price->price,
					(float) $values['price'],
					$values['currency'] ?: null,
					'productForm',
				);

				$this->em->persist($priceHistory);
			}

			if (isset($values['price'])) {
				$price->price = Strings::formatEntityDecimal($values['price']);
			}

			if (Config::load('product.allowRetailPrice') && isset($values['retailPrice'])) {
				$price->retailPrice = Strings::formatEntityDecimal($values['retailPrice']);
			}

			$price->currency = $values['currency'] ?: null;
			$price->vatRate  = $values['vatRate'] ? $this->em->getReference(VatRate::class, $values['vatRate']) : null;

			$product->prices->set($countryId, $price);
			$this->em->persist($price);
		}

		// Delete
		foreach (array_diff_key($currentPrices, $data) as $price) {
			$product->prices->removeElement($price);
			$this->em->remove($price);
		}
	}

	public function savePriceLevelCountriesToProduct(Product $product, array $data): void
	{
		$currentPrices = $product->getPriceLevelCountriesIndexedByGroup();

		foreach ($data as $groupId => $rows) {
			$currentInGroup = $currentPrices[$groupId];

			// Create / Update
			foreach ($rows as $countryId => $row) {
				if (isset($currentInGroup[$countryId])) {
					$price = $currentInGroup[$countryId];
				} else {
					$price = new ProductPriceLevelCountry(
						$product,
						$this->em->getReference(GroupCustomers::class, $groupId),
						$this->em->getReference(Country::class, $countryId),
						$row['currency'],
					);
				}

				if (isset($row['price'])) {
					$price->price = Strings::formatEntityDecimal($row['price']);
				}
				$price->currency = $row['currency'];

				$this->em->persist($price);
			}

			// Delete
			foreach (array_diff_key($currentInGroup, $rows) as $price) {
				$product->priceLevelCountries->removeElement($price);
				$this->em->remove($price);
			}
		}
	}

}