<?php declare(strict_types = 1);

namespace EshopCatalog\FrontModule\Components;

use Core\Model\Helpers\Arrays;
use Core\Model\Sites;
use Core\Model\UI\FrontPresenter;
use Currency\Model\Currencies;
use EshopCatalog\FrontModule\Model\Dao\Product;
use EshopCatalog\FrontModule\Model\Products;
use EshopCatalog\Model\Config;
use EshopOrders\FrontModule\Model\Carts;
use EshopOrders\FrontModule\Model\Dao\AddedCartItem;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use EshopOrders\FrontModule\Model\Event\AddedCartItemEvent;
use EshopOrders\FrontModule\Model\Event\RemovedCartItemEvent;
use EshopOrders\FrontModule\Model\Event\UpdatedCartItemEvent;
use Nette\ComponentModel\IComponent;
use Nette\Utils\ArrayHash;
use Users\Model\Security\User;

class CartAddForm extends BaseControl
{
	public const VIEWTYPE_DEFAULT = 'default';
	public const VIEWTYPE_DETAIL  = 'detail';

	protected Products   $productsService;
	protected ?Product   $product = null;
	protected Carts      $cartsService;
	protected Currencies $currencies;
	protected User       $user;
	protected Sites      $sites;

	public string $viewType = self::VIEWTYPE_DEFAULT;

	/** @var int @persistent */
	public $productId;

	public function __construct(
		Products   $products,
		Carts      $cartsService,
		Currencies $currencies,
		User       $user,
		Sites      $sites
	)
	{
		$this->productsService = $products;
		$this->cartsService    = $cartsService;
		$this->currencies      = $currencies;
		$this->user            = $user;
		$this->sites           = $sites;

		$this->monitor(FrontPresenter::class, function(): void {
			if ($this->productId && !$this->product) {
				$this->setProduct($this->productId);
			}
		});
	}

	public function render(): void
	{
		if (Config::load('allowAddToCartOnlyForLoggedUsers') && !$this->user->isLoggedIn()) {
			return;
		}

		$cart = $this->cartsService->getCurrentCart();
		if (!$cart) {
			return;
		}

		$this->template->viewType = $this->viewType;
		$this->template->product  = $this->product;
		$this->template->cartItem = $cart->getCartItemByProductId($this->product->getId());

		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * ============================== Components
	 */

	protected function createComponentForm(): BaseForm
	{
		$form          = $this->createForm();
		$minimumAmount = $this->product->minimumAmount;
		$maximumAmount = $this->product->getMaximumAmount();
		$site          = $this->sites->getCurrentSite();

		$form->addHidden('productId');
		$form->addHidden('updateQuantity', false);
		$form->addHidden('otherData');
		$form->addText('quantity')->setDefaultValue($minimumAmount)
			->setHtmlAttribute('data-add-to-cart-quantity-input')
			->setHtmlAttribute('data-minimum-amount', $minimumAmount)
			->addRule(
				$form::MIN,
				'eshopCatalogFront.cart.minimumQuantity',
				$minimumAmount > 1 ? $minimumAmount : 0
			);

		$cartProductMaxQuantityByEshop = Config::load('cartProductMaxQuantityByEshop', []) ?? [];

		if (
			(Config::load('pseudoWarehouse') && $this->product->unlimitedQuantity === 0 && $this->product->getQuantity() > 0
				&& (!Config::load('allowWarehouseOverdraft') || !$this->product->getAvailability()->warehouseOverdraft)
				&& (!$maximumAmount || $maximumAmount > $this->product->getQuantity() + $this->product->getQuantityExternal()))
			|| (Arrays::contains($cartProductMaxQuantityByEshop, $site->getIdent()))
		) {
			$maximumAmount = $this->product->getQuantity() + $this->product->getQuantityExternal();
		}

		if ($maximumAmount) {
			$form->getComponent('quantity')->addRule($form::MAX, 'eshopCatalogFront.cart.maximumQuantity', $maximumAmount);
			$form->getComponent('quantity')->setHtmlAttribute('data-max', $maximumAmount);
		}

		if ((Config::load('product.allowNote') && $this->product->getExtraField('addNote'))
			|| $this->product->isAssort) {
			$label = $this->product->getExtraField('noteTitle') ?: $this->translator->translate('eshopCatalogFront.product.addNote');
			$form->addCheckbox('addNote', $label);
			$form->addTextArea('note', $label);
			$form->addHidden('noteTitle', $label);
		}

		$form->onValidate[] = [$this, 'formValidate'];
		$form->onSuccess[]  = [$this, 'formSuccess'];

		return $form;
	}

	public function formValidate(BaseForm $form, ArrayHash $values): void
	{
		if ($form->getErrors()) {
			$this->redrawControl('formErrors');
		}
	}

	public function formSuccess(BaseForm $form, ArrayHash $values): bool
	{
		$this->redrawControl('formErrors');
		$presenter = $this->getPresenter();

		$canAddToCart = $this->productsService->getEr()->createQueryBuilder('p')
			->select('av.canAddToCart')
			->where('p.id = :id')
			->andWhere('p.isDeleted = 0')
			->setParameter('id', $values->productId)
			->innerJoin('p.availability', 'av')
			->getQuery()->setMaxResults(1)->getArrayResult()[0]['canAddToCart'] ?? null;

		if ($canAddToCart) {
			$canAddToCart = (int) $canAddToCart;
		}

		if (!$canAddToCart) {
			$presenter->flashMessageInfo('eshopOrderFront.cart.itemCannotBeAddedToCart');
			$presenter->redrawControl('flashes');

			return false;
		}

		$quantity = (int) $values->quantity;
		$cart     = $this->cartsService->getCurrentCart();
		$cartItem = $cart->getCartItemByProductId((int) $values->productId);
		if ($quantity >= 1) {
			if ($cartItem && $values->updateQuantity) {
				$this->eventDispatcher->dispatch(new UpdatedCartItemEvent(
					(int) $cartItem->getId(),
					(int) $values->quantity,
					(int) $cartItem->quantity,
					(array) $values,
					(int) $cartItem->productId,
				), 'eshopOrders.cartUpdateItem');
				$presenter->payload->cartEvent = 'cartUpdateItem';
			} else {
				$addedItem            = new AddedCartItem;
				$addedItem->quantity  = (int) $values->quantity ?: 1;
				$addedItem->productId = (int) $values->productId;
				$addedItem->moreData  = (array) $values;

				$this->eventDispatcher->dispatch(new AddedCartItemEvent($addedItem), 'eshopOrders.cartAddItem');
				$presenter->payload->cartEvent = 'cartAddItem';
			}
		} else if ($cartItem) {
			$item = new RemovedCartItemEvent($cartItem->getId());
			$this->eventDispatcher->dispatch($item, 'eshopOrders.cartRemoveItem');
			$presenter->payload->cartEvent = 'cartRemoveItem';
		}

		$cartItem = $this->cartsService->getCurrentCart()->getCartItemByProductId((int) $values->productId);
		if ($cartItem) {
			$response = [
				'productId'            => (int) $values->productId,
				'quantity'             => $cartItem->getQuantity(),
				'price'                => $cartItem->getPrice(),
				'priceWithoutVat'      => $cartItem->getPriceWithoutVat(),
				'totalPrice'           => $cartItem->getTotalPrice(),
				'totalPriceWithoutVat' => $cartItem->getTotalPriceWithoutVat(),
				'name'                 => $cartItem->title,
				'image'                => $cartItem->getImage()
					? $this->imagePipe->request($cartItem->getImage()->getFilePath(), Config::load('frontProductDetail.addedToCartPopupImageSize', '80x80'), 'exact')
					: '',
			];
		}
		$presenter->payload->cartItem = $response ?? [];

		$curr                         = $this->currencies->getCurrent();
		$presenter->payload->currency = [
			'symbol'  => $curr->symbol,
			'decimal' => $curr->decimals,
		];

		return true;
	}

	/**
	 * @param int|Product $product
	 */
	public function setProduct($product): self
	{
		if ($product instanceof Product) {
			$this->product   = $product;
			$this->productId = $product->getId();
		} else {
			$this->productId = $product;
		}

		if ($this->productId) {
			$this['form']->setDefaults([
				'productId' => $this->productId,
			]);
		}

		return $this;
	}

}
