<?php declare(strict_types = 1);

namespace EshopCatalog\FrontModule\Components;

use Core\Model\Helpers\Strings;
use Core\Model\UI\BaseControl;
use EshopCatalog\FrontModule\Model\Dao\Product;
use EshopCatalog\FrontModule\Model\Helpers\ShowedProducts;
use EshopCatalog\FrontModule\Model\ProductsFacade;
use EshopCatalog\Model\Config;

class ProductPreview extends BaseControl
{
	protected ProductsFacade        $productsFacade;
	protected ICartAddFormFactory   $cartAddFormFactory;
	protected IVariantsPopupFactory $variantsPopupFactory;
	protected ?Product              $product          = null;
	public string                   $templateFileName = 'ProductPreview';

	public function __construct(
		ProductsFacade        $productsFacade,
		ICartAddFormFactory   $cartAddForm,
		IVariantsPopupFactory $variantsPopupFactory
	)
	{
		$this->productsFacade       = $productsFacade;
		$this->cartAddFormFactory   = $cartAddForm;
		$this->variantsPopupFactory = $variantsPopupFactory;
	}

	public function render(): void
	{
		$groupVariantsInList = Config::load('groupVariantsInList', false) && $this->product->hasVariants();

		if ($groupVariantsInList) {
			$av = $this->product->getAvailability();
			if ($av && !$av->canAddToCart()) {
				foreach ($this->product->variants as $v) {
					if ($v->getAvailability() && $v->getAvailability()->canAddToCart()) {
						$this->product = $v;
						break;
					}
				}
			}
		}

		if ($this->product->getGallery()) {
			if ($groupVariantsInList) {
				$coverImage = $this->product->getGallery()->defaultCover;
			} else {
				$coverImage = $this->product->getGallery()->getCover();
			}
		} else {
			$coverImage = null;
		}

		$this->template->product             = $this->product;
		$this->template->groupVariantsInList = $groupVariantsInList;
		$this->template->coverImage          = $coverImage;
		ShowedProducts::addShowProduct($this->product->getId());
		$this->template->render($this->getTemplateFile($this->templateFileName, 'ProductPreview'));
	}

	protected function createComponentCartAddForm(): CartAddForm
	{
		$control = $this->cartAddFormFactory->create();
		$control->setProduct($this->product);

		return $control;
	}

	protected function createComponentVariantsPopup(): VariantsPopup
	{
		if (!$this->product) {
			$do = (string) $this->getPresenter()->getParameter('do');

			if (Strings::startsWith($do, 'list-productPreview')) {
				preg_match('!\d+!', (string) $this->getPresenter()->getParameter('do'), $matches);
				if (isset($matches[0])) {
					$this->product = $this->productsFacade->getProduct((int) $matches[0]);
				}
			} elseif ($this->getPresenter()->getParameter('id')) {
				$this->product = $this->productsFacade->getProduct((int) $this->getPresenter()->getParameter('id'));
			}
		}

		return $this->variantsPopupFactory->create($this->product);
	}

	public function setProduct(Product $product): self
	{
		$this->product = $product;

		return $this;
	}

	public function setProductById(int $id): self
	{
		$product = $this->productsFacade->getProduct($id);

		if ($product) {
			$this->product = $product;
		}

		return $this;
	}

}