<?php declare(strict_types = 1);

namespace EshopCatalog\FrontModule\Components;

use Core\Model\Event\ControlEvent;
use Core\Model\Helpers\Strings;
use Core\Model\Parameters;
use Core\Model\UI\BaseControl;
use Core\Model\UI\FrontPresenter;
use Currency\Model\Exchange;
use EshopCatalog\FrontModule\Model\Categories;
use EshopCatalog\FrontModule\Model\Dao\FilterGroup;
use EshopCatalog\FrontModule\Model\Event\FilterEvent;
use EshopCatalog\FrontModule\Model\Event\FilterLinkEvent;
use EshopCatalog\FrontModule\Model\FeatureProducts;
use EshopCatalog\FrontModule\Model\FilterService;
use EshopCatalog\FrontModule\Model\Products;
use EshopCatalog\FrontModule\Model\ProductsFacade;
use EshopCatalog\Model\Config;
use EshopCatalog\Model\Entities\Feature;
use EshopCatalog\Model\Helpers\VariantsHelper;
use Exception;
use Nette\Application\UI\InvalidLinkException;
use Nette\ComponentModel\IComponent;

class ProductsFilter extends BaseControl
{
	/** @var int[] */
	public array $productIds = [];

	/** @var int[] */
	protected array $productIdsAfterFilter = [];

	protected ?array $resultAfterFilter = null;

	/** @var int[] */
	public array $productIdsNoPriceFilter = [];

	protected array $categories  = [];
	public ?string  $searchQuery = null;

	protected FeatureProducts $featureProducts;
	protected Products        $productsService;
	protected FilterService   $filterService;
	protected Categories      $categoriesService;
	protected ProductsFacade  $productsFacade;
	protected Exchange        $exchange;
	protected VariantsHelper  $variantsHelper;

	/** @var callable[] */
	public array $onFilter = [];

	protected array  $sortValues      = [];
	public ?array    $allowedFeatures = null;
	protected ?array $cFilters        = null;
	protected ?array $cMinMax         = null;
	protected ?array $filterValues    = null;

	/** @persistent */
	public ?array $filter = [];

	/** @persistent */
	public array $fr = [];

	/** @var int|float|string|null @persistent */
	public $min;

	/** @var int|float|string|null @persistent */
	public $max;

	/** @var string|null @persistent */
	public $sort;

	public function __construct(
		FeatureProducts $featureProducts,
		Products        $products,
		FilterService   $filterService,
		Categories      $categories,
		ProductsFacade  $productsFacade,
		Exchange        $exchange,
		VariantsHelper  $variantsHelper
	)
	{
		$this->featureProducts   = $featureProducts;
		$this->productsService   = $products;
		$this->filterService     = $filterService;
		$this->categoriesService = $categories;
		$this->productsFacade    = $productsFacade;
		$this->exchange          = $exchange;
		$this->variantsHelper    = $variantsHelper;
		$this->sortValues        = $this->getSortValues();

		$this->monitor(FrontPresenter::class, function(FrontPresenter $presenter): void {
			$do = $presenter->getParameter('do');

			if (!$do || !(Strings::contains($do, 'cartAddForm') || Strings::contains($do, 'productPreview'))) {
				if ($presenter->isAjax()) {
					if ($do && Strings::contains($do, 'formSubmit')) {
						$this->formSubmit();
					}
					$this->eventDispatcher->dispatch(new FilterEvent($this->getFilterValues()), ProductsFilter::class . '::onFilter');
					$this->redrawControl('filters');
				}

				$this->eventDispatcher->addListener(ProductsFilter::class . '::onFilter', function() {
					$this->redrawControl('filters');
				});
			}
		});
	}

	public function getSortValues(): array
	{
		return [
			'recommended',
			'p.price+ASC',
			'p.price+DESC',
			'p.created+DESC',
		];
	}

	public function render(): void
	{
		$filters      = $this->applyFilters();
		$minMax       = $this->getMinMax();
		$filterActive = false;
		foreach ($filters as $group) {
			if ($group->hasActive()) {
				$filterActive = true;
				break;
			}
		}

		if (Config::load('productPreview.basePriceWithoutVat', false)) {
			if (isset($minMax['min'])) {
				$minMax['min'] = round($minMax['min'] / (1 + ((float) $minMax['minVat']) / 100), 2);
			}
			if (isset($minMax['max'])) {
				$minMax['max'] = round($minMax['max'] / (1 + ((float) $minMax['maxVat']) / 100), 2);
			}
		}

		$this->template->filters       = $filters;
		$this->template->minMax        = $minMax;
		$this->template->currentMinMax = [
			'min' => $this->min ?: '',
			'max' => $this->max ?: '',
		];
		$this->template->currentSort   = $this->sort;
		$this->template->sortValues    = $this->getSortValues();

		$this->template->filterActive = $filterActive || $this->min || $this->max || !empty($this->fr);

		try {
			$tmp                       = $this->getPresenter()->template->breadcrumb;
			$this->template->resetLink = $this->searchQuery ? $this->link('this', ['filter' => []]) : end($tmp)->link;
		} catch (Exception $e) {
			$presenter = $this->getPresenter();
			$args      = $presenter->getParameters();
			unset($args['activeNavigation']);
			unset($args['action']);
			unset($args['do']);
			$this->template->resetLink = $presenter->link(':' . $presenter->getName() . ':' . $presenter->getAction(), $args);
		}

		$this->template->render($this->getTemplateFile());
	}

	public function createLink(string $type, string $key, string $value): string
	{
		$result = null;
		if (Parameters::load('eshopAdvancedFeature.allowVirtualCategories')) {
			$event = new FilterLinkEvent($this->filterValues ?? [], $this->categories, $type, $key, $value);
			$this->eventDispatcher->dispatch($event, __CLASS__ . '::createFilterLink');

			if ($event->result) {
				$result = $event->result;
			}
		}

		return $result ?: $this->link('set!', func_get_args());
	}

	public function createSortLink(?string $sort): string
	{
		$result = null;
		if (Parameters::load('eshopAdvancedFeature.allowVirtualCategories')) {
			$event = new FilterLinkEvent($this->filterValues ?? [], $this->categories, 'sort', 'sort', $sort);
			$this->eventDispatcher->dispatch($event, __CLASS__ . '::createSortLink');

			if ($event->result) {
				$result = $event->result;
			}
		}

		return $result ?: $this->link('sort!', func_get_args());
	}

	/*******************************************************************************************************************
	 * ============================== Handle
	 */

	/**
	 * @param int|string $t
	 * @param int|string $k
	 * @param int|string $v
	 *
	 * @crossOrigin
	 *
	 * @throws InvalidLinkException
	 */
	public function handleSet($t, $k, $v): void
	{
		if ($t) {
			$this->filter['f' . $t][$k][] = $v;
		}

		if ($this->getPresenter()->isAjax()) {
			$this->eventDispatcher->dispatch(new FilterEvent($this->getFilterValues()), ProductsFilter::class . '::onFilter');
			$this->getPresenter()->payload->url = $this->link('this');
		}
	}

	/**
	 * @param int|string $min
	 * @param int|string $max
	 *
	 * @throws InvalidLinkException
	 */
	public function handlePrice($min, $max): void
	{
		$this->min = $this->getMinMax()['min'] == $min ? null : $min;
		$this->max = $this->getMinMax()['max'] == $max ? null : $max;

		if ($this->getPresenter()->isAjax()) {
			$this->eventDispatcher->dispatch(new FilterEvent($this->getFilterValues()), ProductsFilter::class . '::onFilter');
			$this->getPresenter()->payload->url = $this->link('this');
		}
	}

	public function handleSort(string $sort): void
	{
		if ($sort === 'recommended') {
			$sort = null;
		}

		$this->sort = $sort;

		if ($this->getPresenter()->isAjax()) {
			$this->eventDispatcher->dispatch(new FilterEvent($this->getFilterValues()), ProductsFilter::class . '::onFilter');
			$this->getPresenter()->payload->url = $this->link('this');
		}
	}

	public function handleFormSubmit(): void
	{
		$this->formSubmit();
		if ($this->getPresenter()->isAjax()) {
			$this->eventDispatcher->dispatch(new FilterEvent($this->getFilterValues()), ProductsFilter::class . '::onFilter');
			$this->getPresenter()->payload->url = $this->link('this');
		}
	}

	protected function formSubmit(): void
	{
		$presenter   = $this->getPresenter();
		$httpRequest = $presenter->getHttpRequest();
		$this->resetFilter();

		$catIds = $presenter->getParameter('id');
		if ($catIds && is_array($catIds)) {
			$this->filter['c'] = implode('|', $catIds);
		}

		$features = $httpRequest->getPost('feature');
		if ($features) {
			foreach ($features as $g => $vals) {
				if (is_string($vals) && $vals === 'on') {
					continue;
				}
				$this->filter['ff'][$g] = implode('|', $vals);
			}
		}

		foreach ($httpRequest->getPost('range') ?: [] as $k => $v) {
			if ($v['min'] || $v['max']) {
				$this->fr[$k] = implode('|', $v);
			} else {
				unset($this->fr[$k]);
			}
		}

		$manufacturers = $httpRequest->getPost('manu');
		if ($manufacturers) {
			foreach ($manufacturers as $g => $vals) {
				if (is_string($vals) && $vals === 'on') {
					continue;
				}
				$this->filter['fm'][$g] = implode('|', $vals);
			}
		}

		$tags = $httpRequest->getPost('tags') ?: $httpRequest->getPost('tag');
		if ($tags) {
			foreach ($tags as $g => $vals) {
				if (is_string($vals) && $vals === 'on') {
					continue;
				}
				$this->filter['ft'][$g] = implode('|', $vals);
			}
		}

		$priceRange = $httpRequest->getPost('priceRange');
		if ($priceRange) {
			if (isset($priceRange['min']) && $this->getMinMax()['min'] != $priceRange['min']) {
				$this->min = $priceRange['min'];
			}
			if (isset($priceRange['max']) && $this->getMinMax()['max'] != $priceRange['max']) {
				$this->max = $priceRange['max'];
			}
		}

		$sort = $httpRequest->getPost('sort');
		if ($sort) {
			$this->sort = $sort;
		}
	}

	/*******************************************************************************************************************
	 * ============================== Get / Set
	 */

	public function setCategories(array $categories): self
	{
		$this->categories                     = array_map(static fn($c) => $c ? $c->getId() : null, $categories);
		$this->categories                     = array_filter($this->categories, 'strlen');
		FilterService::$prepareDataCategories = $this->categories;
		$this->productIds                     = $this->productsService->getProductsIdInCategory($this->categories, 0, null, $this->sort);
		$this->loadFilters();

		return $this;
	}

	public function setSearchQuery(string $search): self
	{
		$this->searchQuery = $search;
		$this->productIds  = $this->productsService->getProductsIdBySearch($search, 0, null, $this->sort);
		$this->loadFilters();

		return $this;
	}

	protected function filterPriceCheck(array $activeFilter, array $productData): bool
	{
		$pr    = $activeFilter['priceRange'] ?? null;
		$price = $productData['price'];

		return !$pr
			|| (!isset($pr['min']) || $price >= $pr['min']) && (!isset($pr['max']) || $price <= $pr['max']);
	}

	protected function filterRangeCheck(array $activeFilter, array $productData): bool
	{
		foreach ($activeFilter['range'] as $k => $v) {
			$df = $productData['df'][$k] ?? null;
			if (!$df) {
				return false;
			}

			if (($v[0] && max($df) < $v[0]) || ($v[1] && min($df) > $v[1])) {
				return false;
			}
		}

		return true;
	}

	protected function loadFilters(): void
	{
		$this->eventDispatcher->dispatch(new ControlEvent($this), ProductsFilter::class . '::beforeLoadFilters');

		$filter = $this->getFilterValues();

		$filterNoPrice = $filter;
		unset($filterNoPrice['priceRange']);
		if (!empty($this->categories)) {
			$this->productIdsNoPriceFilter = $this->productsService->getProductsIdInCategory($this->categories, null, null, null, $filterNoPrice);
		} else if ($this->searchQuery) {
			$this->productIdsNoPriceFilter = $this->productsService->getProductsIdBySearch($this->searchQuery, null, null, null, $filterNoPrice);
		}

		$this->applyFilters();
	}

	/**
	 * @return FilterGroup[]
	 */
	public function applyFilters(): ?array
	{
		if ($this->cFilters) {
			return $this->cFilters;
		}

		$activeFilters = $this->getFilterValues();
		$filterGroups  = [];

		$tagGroup              = new FilterGroup;
		$tagGroup->type        = 'tag';
		$tagGroup->id          = 0;
		$tagGroup->name        = $this->t('eshopCatalogFront.filter.tags');
		$tagGroup->items       = $this->filterService->getTags($this->productIds);
		$filterGroups['tag_0'] = $tagGroup;

		$manuGroup              = new FilterGroup;
		$manuGroup->type        = 'manu';
		$manuGroup->id          = 0;
		$manuGroup->name        = $this->t('eshopCatalogFront.filter.manufacturer');
		$manuGroup->items       = $this->filterService->getManufacturers($this->productIds);
		$filterGroups['manu_0'] = $manuGroup;

		$availableFeatures = [];
		if (!empty($this->categories)) {
			foreach ($this->categories as $catId) {
				$cat = $this->categoriesService->get((int) $catId);

				if ($cat) {
					foreach ($this->filterService->getCategoryFilters((int) $catId, $cat->filtersFromParent ? true : false) as $fea) {
						if (!isset($availableFeatures[$fea->id]) && $fea->name) {
							$availableFeatures[$fea->id] = $fea;
						}
					}
				}
			}
		} else {
			$availableFeatures = $this->filterService->getFeatures($this->productIds);
			uasort($availableFeatures, static fn($a, $b) => $a->position <=> $b->position);
		}

		$productsFilters = $this->getProductsCountInFilter()['filters'];
		$products        = $this->getProductsCountInFilter()['p'];
		$productsDf      = $this->getProductsCountInFilter()['df'];

		if ($this->allowedFeatures) {
			$tmpF = [];
			foreach ($this->allowedFeatures as $v) {
				if (isset($availableFeatures[$v])) {
					$tmpF[$v] = $availableFeatures[$v];
				}
			}

			$availableFeatures = $tmpF;
			$tmpF              = null;
		}
		foreach ($availableFeatures as $k => $v) {
			if ($v->valueType === Feature::TYPE_CHECK) {
				$filterGroups['feature_' . $k] = $v;
			} else {
				$tmp      = $productsDf[$k] ?? null;
				$groupKey = $v->valueType . '_' . $k;

				$filterGroups[$groupKey] = $v;
				if ($tmp) {
					$filterGroups[$groupKey]->setMinMaxValue($tmp['min'], $tmp['max']);
				}

				$af = $activeFilters[$v->valueType][$k] ?? null;
				if ($af) {
					$filterGroups[$groupKey]->setMinMaxCurrentValue(...$af);
				}
			}
		}
		$availableFeatures = null;

		foreach ($filterGroups as $kg => $feature) {
			foreach ($feature->items as $ki => $item) {
				$count = count($productsFilters[$feature->type . '_' . $ki]);

				if ($count) {
					$item->productsCountInit = $count;
				} else {
					unset($feature->items[$ki]);
				}
			}
		}

		$prodsForFilter   = [];
		$filteredProducts = [];
		if (isset($activeFilters['tag']) || isset($activeFilters['manu']) || isset($activeFilters['feature'])) {
			foreach ($activeFilters as $afGroup => $afGroupValue) {
				foreach ($afGroupValue as $afGroupValuesKey => $afValues) {
					if (in_array($afGroup, ['tag', 'manu'])) {
						$afValues = $afGroupValue;
					}

					foreach ($afValues as $value) {
						if (isset($productsFilters[$afGroup . '_' . $value])) {
							$prodsForFilter[$afGroup . $afGroupValuesKey] = array_merge($prodsForFilter[$afGroup . $afGroupValuesKey] ?? [], $productsFilters[$afGroup . '_' . $value] ?? []);
						}

						$group = $filterGroups[$afGroup . '_' . $afGroupValuesKey] ?? null;
						if ($group) {
							foreach ($group->items as $itemK => $item) {
								if ($itemK == $value) {
									$item->isActive = true;
									continue;
								}

								$item->productsCount = count($productsFilters[$afGroup . '_' . $itemK] ?? []);
							}
						}
					}
				}
			}

			if (!empty($prodsForFilter)) {
				$prodsForFilter = count($prodsForFilter) > 1 ? array_intersect(...array_values($prodsForFilter)) : array_values($prodsForFilter)[0];
			}
		} else {
			$prodsForFilter = $this->productIds;
		}

		if ($prodsForFilter) {
			$prodsForFilter = array_unique($prodsForFilter);
		}

		if (isset($activeFilters['priceRange']) || isset($activeFilters['range'])) {
			foreach ($prodsForFilter as $k => $prodId) {
				$prod = $products[$prodId] ?? null;

				if (!$prod || !$this->filterPriceCheck($activeFilters, $prod) || !$this->filterRangeCheck($activeFilters, $prod)) {
					unset($prodsForFilter[$k]);
					continue;
				}
			}
		}

		if (count($prodsForFilter) === count($this->productIds)) {
			foreach ($filterGroups as $kg => $feature) {
				foreach ($feature->items as $ki => $item) {
					$item->productsCount = count($productsFilters[$feature->type . '_' . $ki]);
				}
			}
			$this->productIdsAfterFilter = $this->productIds;
		} else {
			foreach ($prodsForFilter as $prodId) {
				$prod = $products[$prodId] ?? null;
				if (!$prod) {
					continue;
				}

				$filteredProducts[] = $prodId;

				foreach ($prod['filters'] as $k => $v) {
					$group = $filterGroups[$k] ?? null;
					if ($group) {
						if (is_array($v)) {
							foreach ($v as $v2) {
								$item = $group->items[$v2] ?? null;

								if ($item) {
									$item->productsCount++;
								}
							}
						} else {
							$item = $group->items[$v] ?? null;

							if ($item) {
								$item->productsCount++;
							}
						}
					}
				}
			}

			$this->productIdsAfterFilter = array_unique($filteredProducts);
		}

		$this->cFilters = $filterGroups;

		return $this->cFilters;
	}

	public function getMinMax(): ?array
	{
		if (!$this->cMinMax && $this->productIdsNoPriceFilter) {
			$min    = $this->getProductsCountInFilter()['priceMin'];
			$minVat = $this->getProductsCountInFilter()['priceMinVatRate'] ?? 21;
			$max    = $this->getProductsCountInFilter()['priceMax'];
			$maxVat = $this->getProductsCountInFilter()['priceMaxVatRate'] ?? 21;

			$this->cMinMax = [
				'min'    => $this->exchange->change((float) $min),
				'max'    => $this->exchange->change((float) $max),
				'minVat' => $minVat,
				'maxVat' => $maxVat,
			];
		}

		return $this->cMinMax;
	}

	/**
	 * Vrátí ID produktů které odpovídají filtru
	 */
	public function getProductIdsAfterFilter(): array
	{
		if ($this->resultAfterFilter === null) {
			$products = $this->productIdsAfterFilter
				? array_intersect($this->productIds, $this->productIdsAfterFilter)
				: ($this->getFilterValues() ? [] : $this->productIds);

			if (Config::load('groupVariantsInList', false)) {
				$allVariants = $this->variantsHelper->getBasicByProduct();

				$tmp          = [];
				$usedVariants = [];

				foreach ($products as $k => $id) {
					$v = $allVariants[$id] ?? null;

					if (!$v) {
						$tmp[$id] = $id;
					} else if (!isset($usedVariants[$v['variantId']])) {
						$tmp[$id]                      = $id;
						$usedVariants[$v['variantId']] = $id;
					} else if ($v['isDefault'] === 1) {
						$tmp[$usedVariants[$v['variantId']]] = $id;
					}
				}

				$this->resultAfterFilter = array_values($tmp);
			} else {
				$this->resultAfterFilter = $products;
			}
		}

		return $this->resultAfterFilter;
	}

	public function getFilterValues(): ?array
	{
		if ($this->filterValues) {
			return $this->filterValues;
		}

		$return = [];

		if (isset($this->filter['c'])) {
			$return['categories'] = explode('|', $this->filter['c']);
		}

		if (isset($this->filter['ff'])) {
			foreach ($this->filter['ff'] as $group => $vals) {
				$return['feature'][$group] = explode('|', $this->filter['ff'][$group]);
			}
		}

		if (isset($this->fr)) {
			foreach ($this->fr as $group => $vals) {
				$return['range'][$group] = explode('|', $this->fr[$group]);
			}
		}

		if (isset($this->filter['fm'])) {
			foreach ($this->filter['fm'] as $group => $vals) {
				$return['manu'] = explode('|', $this->filter['fm'][$group]);
			}
		}

		if (isset($this->filter['ft'])) {
			foreach ($this->filter['ft'] as $group => $vals) {
				$return['tag'] = explode('|', $this->filter['ft'][$group]);
			}
		}

		if ($this->min) {
			$return['priceRange']['min'] = $this->min;
		}

		if ($this->max) {
			$return['priceRange']['max'] = $this->max;
		}

		if (isset($return['priceRange']['min']) && $return['priceRange']['min'] > 0) {
			if (Config::load('productPreview.basePriceWithoutVat', false)) {
				$return['priceRange']['min'] = ((float) $return['priceRange']['min'] / 100) * 121;
			}

			$return['priceRange']['min'] = $this->exchange->change((float) $return['priceRange']['min'], 'default', 'current');
		}

		if (isset($return['priceRange']['max']) && $return['priceRange']['max'] > 0) {
			if (Config::load('productPreview.basePriceWithoutVat', false)) {
				$return['priceRange']['max'] = ((float) $return['priceRange']['max'] / 100) * 121;
			}

			$return['priceRange']['max'] = $this->exchange->change((float) $return['priceRange']['max'], 'default', 'current');
		}

		if ($this->sort) {
			$this->sort     = urldecode($this->sort);
			$this->sort     = str_replace(' ', '+', $this->sort);
			$return['sort'] = $this->sort;
		}

		$this->filterValues = $return;

		return $this->filterValues;
	}

	protected function resetFilter(): void
	{
		$this->filter = null;
		$this->min    = null;
		$this->max    = null;
		$this->sort   = null;
	}

	protected function getProductsCountInFilter(): ?array
	{
		$key = $this->categories ? 'c' . serialize($this->categories) : $this->searchQuery;

		if (!$key) {
			return [];
		}

		return $this->filterService->getProductsData($this->productIds, $key);
	}

	public function checkDisableIndexing(): bool
	{
		$filters = $this->getFilterValues();

		if (!empty($filters)) {
			return true;
		}

		return false;
	}

	public function checkDisableIndexingAdvanced(): bool
	{
		$filters = $this->getFilterValues();

		if (
			isset($filters['priceRange'])
			|| isset($filters['range'])
			|| isset($filters['tag'])
			|| (isset($filters['manu']) && count($filters['manu']) >= 2)
		) {
			return true;
		}

		foreach ($filters['feature'] ?? [] as $feature => $values) {
			if (count($values) >= 2) {
				return true;
			}
		}

		return false;
	}

}
