<?php declare(strict_types = 1);

namespace EshopCatalog\FrontModule\Model;

use Core\Model\Entities\ExtraField;
use Core\Model\Helpers\BaseFrontEntityService;
use Core\Model\Lang\DefaultLang;
use EshopCatalog\Model\Config;
use EshopCatalog\Model\Entities\Feature;
use EshopCatalog\Model\Entities\FeatureValue;
use EshopCatalog\Model\Helpers\FeaturesHelper;
use Nette\Caching\Cache;
use Throwable;

class Features extends BaseFrontEntityService
{
	public const CACHE_NAMESPACE = 'eshopCatalogFeatures';

	protected $entityClass = Feature::class;

	protected DefaultLang  $defaultLang;
	protected CacheService $cacheService;
	public FeaturesHelper  $featuresHelper;

	protected array  $cacheDep       = [
		Cache::TAGS   => ['features'],
		Cache::EXPIRE => '2 weeks',
	];
	protected ?array $cFeatures      = null;
	protected ?array $cFeatureValues = null;

	public function __construct(
		CacheService   $cacheService,
		DefaultLang    $defaultLang,
		FeaturesHelper $featuresHelper
	)
	{
		$this->cacheService   = $cacheService;
		$this->defaultLang    = $defaultLang;
		$this->featuresHelper = $featuresHelper;
	}

	public function getPublishedFeatures(): array
	{
		$features = [];
		foreach ($this->getEr()->createQueryBuilder('f')
			         ->select('f.id, ft.name')
			         ->join('f.featureTexts', 'ft')
			         ->where('f.isPublished = 1')->andWhere('ft.lang = :lang')
			         ->setParameter('lang', $this->defaultLang->locale)
			         ->orderBy('f.position')
			         ->getQuery()->getScalarResult() as $row) {
			$features[$row['id']] = $row['name'];
		}

		return $features;
	}

	/**
	 * @param string|int $id
	 */
	public function getFeatureById($id): array
	{
		if (!$this->cFeatures[$id]) {
			$locale = $this->translator->getLocale();
			$key    = 'features_' . $locale;

			if ($this->cFeatures === null) {
				$this->cFeatures = $this->cacheService->defaultCache->load($key, function(&$dep) use ($locale) {
					$dep = $this->cacheDep;

					$select = 'f.id, ft.name, f.position, f.unit, f.type, f.decimals, f.useForVariantDiff, f.valueType';
					if (Config::load('features.allowDescription')) {
						$select .= ', ft.productTabTitle';
					}

					$data = [];
					foreach ($this->em->getRepository(Feature::class)->createQueryBuilder('f')
						         ->select($select)
						         ->andWhere('f.isPublished = 1')
						         ->join('f.featureTexts', 'ft', 'WITH', 'ft.lang = :lang')
						         ->setParameter('lang', $locale)->getQuery()->getArrayResult() as $row) {
						$data[$row['id']] = $row;
					}

					return $data;
				});
			}
		}

		return $this->cFeatures[$id];
	}

	public function getValuesByFeatureId(int $id): array
	{
		if (!isset($this->cFeatures[$id]['values'])) {
			$key = 'featureIdValues';

			$data = $this->cacheService->defaultCache->load($key, function(&$dep) {
				$dep                = $this->cacheDep;
				$dep[Cache::TAGS][] = 'featureValues';

				$data = [];

				foreach ($this->em->getRepository(FeatureValue::class)->createQueryBuilder('fv')
					         ->select('fv.id as id, IDENTITY(fv.feature) as featureId')
					         ->andWhere('fv.isPublished = 1')->orderBy('fv.position')
					         ->getQuery()->getArrayResult() as $row) {
					$data[$row['featureId']][] = $row['id'];
				}

				return $data;
			});

			foreach ($data as $k => $v) {
				foreach ($v as $vId) {
					$this->cFeatures[$k]['values'][$vId] = $this->getFeatureValueById($vId);
				}
			}
		}

		return $this->cFeatures[$id]['values'] ?: [];
	}

	/**
	 * @param string|int $id
	 *
	 * @throws Throwable
	 */
	public function getFeatureValueById($id): array
	{
		if (!$this->cFeatureValues[$id]) {
			$locale = $this->defaultLang->locale;
			$key    = 'featureValues_' . $locale;

			if ($this->cFeatureValues === null) {
				$this->cFeatureValues = $this->cacheService->defaultCache->load($key, function(&$dep) use ($locale) {
					$dep                = $this->cacheDep;
					$dep[Cache::TAGS][] = 'featureValues';

					$select = 'fv.id as id, fv.position, fvt.name as name, fvt.rodM, fvt.rodZ, fvt.rodS, IDENTITY(fv.feature) as featureId, 
						f.valueType, fv.value1';
					if (Config::load('features.allowImages')) {
						$select .= ', fv.image';
					}
					if (Config::load('features.allowDescription')) {
						$select .= ', fv.showAsTag, fv.moreLink, fv.tagTextColor, fv.tagBgColor, fvt.tagText, fvt.shortDescription, fvt.longDescription';
					}

					$data = [];
					foreach ($this->em->getRepository(FeatureValue::class)->createQueryBuilder('fv')
						         ->select($select)
						         ->andWhere('fv.isPublished = 1')
						         ->join('fv.featureValueTexts', 'fvt', 'WITH', 'fvt.lang = :lang ')
						         ->join('fv.feature', 'f')
						         ->orderBy('fv.position')
						         ->setParameter('lang', $locale)->getQuery()->getArrayResult() as $row) {
						if (!isset($row['extraFields'])) {
							$row['extraFields'] = [];
						}

						if ($row['valueType'] === Feature::VALUE_TYPE_COLOR && !empty($row['value1'])) {
							$row['extraFields'] += $this->featuresHelper->parseValueColors((string) $row['value1']);
						}

						$data[$row['id']] = $row;
					}


					foreach ($this->em->getRepository(ExtraField::class)->createQueryBuilder('ef')
						         ->select('ef.sectionKey, ef.key, ef.value')
						         ->where('ef.sectionName = :secName')
						         ->andWhere('ef.lang IS NULL OR ef.lang = :lang')
						         ->setParameters([
							         'lang'    => $this->translator->getLocale(),
							         'secName' => FeatureValue::EXTRA_FIELD_SECTION,
						         ])->getQuery()->getScalarResult() as $row) {
						$data[$row['sectionKey']]['extraFields'][$row['key']] = $row['value'];
					}

					return $data;
				});
			}
		}

		return $this->cFeatureValues[$id] ?? [];
	}

}
