<?php declare(strict_types = 1);

namespace EshopCatalog\FrontModule\Model;

use Core\Model\Countries;
use Core\Model\Images\Image;
use Doctrine\ORM\NonUniqueResultException;
use Doctrine\ORM\Query\Expr\Join;
use EshopCatalog\Model\Config;
use EshopCatalog\Model\Entities\Availability;
use EshopCatalog\Model\Entities\CategoryProduct;
use EshopCatalog\Model\Entities\ProductSpedition;
use EshopCatalog\Model\Entities\ProductVariant;
use EshopCatalog\Model\Entities\RelatedProduct;
use Gallery\FrontModule\Model\Dao\Album;
use Nette\Application\LinkGenerator;
use Core\Model\Sites;
use EshopCatalog\FrontModule\Model\Event\ProductsEvent;
use Core\Model\Event\EventDispatcher;
use Gallery\FrontModule\Model\Albums;
use Nette\Application\UI\InvalidLinkException;
use Nette\Caching\Cache;
use Contributte\Translation\Translator;
use Throwable;
use Users\Model\Http\UserStorage;
use EshopCatalog\FrontModule\Model\Dao;

class ProductsFacade
{
	public const MODE_ESHOP    = 'eshop';
	public const MODE_CHECKOUT = 'checkout';

	protected static string $mode         = self::MODE_ESHOP;
	public static ?string   $forceLocale  = null;
	public static ?bool     $cacheEnabled = true;

	protected Translator              $translator;
	protected CacheService            $cacheService;
	public Products                   $productsService;
	protected EventDispatcher         $eventDispatcher;
	public LinkGenerator              $linkGenerator;
	protected UserStorage             $userStorage;
	protected Albums                  $albumsService;
	protected Manufacturers           $manufacturersService;
	protected Tags                    $tagsService;
	protected Sites                   $sitesService;
	public Categories                 $categoriesService;
	protected DynamicFeaturesProducts $dynamicFeaturesProducts;
	protected Countries               $countries;

	protected Packages $packages;

	protected AvailabilityService $availabilityService;

	/** @var Dao\Product[] */
	protected array $cProducts;

	/** @var Dao\Product[] */
	protected array $cVariants;

	public array        $cFreashLoaded = [];
	public static array $allLoadedIds  = [];
	public static bool  $loadFeatures  = true;
	public static bool  $loadTags      = true;

	public function __construct(
		Translator              $translator,
		CacheService            $cacheService,
		Products                $products,
		EventDispatcher         $eventDispatcher,
		UserStorage             $userStorage,
		Albums                  $albums,
		Manufacturers           $manufacturers,
		Tags                    $tags,
		Sites                   $sites,
		Categories              $categories,
		DynamicFeaturesProducts $dynamicFeaturesProducts,
		Countries               $countries,
		Packages                $packages,
		AvailabilityService     $availabilityService
	)
	{
		$this->translator              = $translator;
		$this->cacheService            = $cacheService;
		$this->productsService         = $products;
		$this->eventDispatcher         = $eventDispatcher;
		$this->userStorage             = $userStorage;
		$this->albumsService           = $albums;
		$this->manufacturersService    = $manufacturers;
		$this->tagsService             = $tags;
		$this->sitesService            = $sites;
		$this->categoriesService       = $categories;
		$this->dynamicFeaturesProducts = $dynamicFeaturesProducts;
		$this->countries               = $countries;
		$this->packages                = $packages;
		$this->availabilityService     = $availabilityService;
	}

	public function loadBase(array $ids): array
	{
		$whereIds = [];
		$loaded   = [];
		$result   = [];
		$locale   = $this->getLocale();

		foreach ($ids as $k => $id) {
			if (isset($this->cProducts[$id])) {
				$loaded[$id] = $this->cProducts[$id];
				unset($ids[$k]);
			}
		}

		if (self::$cacheEnabled) {
			$keys = [];
			foreach ($ids as $id) {
				$keys[] = 'product/' . $id . '/' . $locale;
			}

			// Načtení z cache
			if ($keys) {
				foreach ($this->cacheService->productCache->bulkLoad($keys) as $key => $product) {
					$tmp = explode('/', $key);
					$id  = $tmp[1];

					if ($product) {
						$result[$id] = $product;
					} else {
						$whereIds[] = $id;
					}
				}
			}
		} else {
			$whereIds = array_values($ids);
		}

		// Vytvoření získání produktů co nejsou v cache
		if (!empty($whereIds)) {
			$query = (new ProductQuery($locale))
				->withTexts()
				->withVatRate()
				->loadSites()
				->withVariantParent()
				->byId($whereIds);

			$rootCats = $this->categoriesService->getAllRootIds();

			$cats = [];
			foreach ($this->productsService->em->createQueryBuilder()
				         ->select('IDENTITY(cp.product) as product, IDENTITY(cp.category) as category')
				         ->from(CategoryProduct::class, 'cp')
				         ->where('cp.product IN (:ids)')
				         ->setParameter('ids', $whereIds)
				         ->getQuery()->getScalarResult() as $cat) {
				$cats[$cat['product']][] = $cat['category'];
			}

			$qb = $query->getQueryBuilder($this->productsService->getEr());
			$qb->groupBy('p.id');

			foreach ($this->productsService->customQB as $cqb) {
				$cqb($qb);
			}

			$disabledSpeds = [];
			foreach ($this->productsService->em->getRepository(ProductSpedition::class)->createQueryBuilder('s')
				         ->where('s.product IN (:ids)')->setParameter('ids', $whereIds)
				         ->andWhere('s.speditionDisabled = 1')
				         ->getQuery()->getScalarResult() as $row) {
				$disabledSpeds[$row['s_product_id']][] = $row['s_spedition_id'];
			}

			if (Config::load('product.allowCountryOfOrigin')) {
				$qb->addSelect('IDENTITY(p.countryOfOrigin) as countryOfOrigin');
			}

			if (Config::load('product.allowPackage', false)) {
				$qb->addSelect('IDENTITY(p.package) as packageId');
			}

			$qbResult   = $qb->getQuery()->getArrayResult();
			$idsForLink = [];
			foreach ($qbResult as $row) {
				$idsForLink[] = $row[0]['id'];
			}

			self::$allLoadedIds = $idsForLink;

			$featureValuesIds = $this->productsService->featureProductsService->getFeatureValuesIdsForProducts($idsForLink);
			$dynamicFeatures  = $this->dynamicFeaturesProducts->getDaoByProducts($idsForLink);
			$extraFields      = $this->productsService->getExtraFieldsByIds($idsForLink);

			foreach ($qbResult as $row) {
				$row[0]['categories']      = $cats[$row[0]['id']] ?? [];
				$tmp                       = $this->productsService->normalizeHydrateArray($row);
				$tmp['extraFields']        = $extraFields[$tmp['id']] ?: [];
				$tmp['disabledSpeditions'] = $disabledSpeds[$row[0]['id']] ?? [];
				$tmp['dynamicFeatures']    = $dynamicFeatures[$tmp['id']] ?? [];

				if ($row['countryOfOrigin']) {
					$tmp['countryOfOrigin'] = $this->countries->getDao()[$row['countryOfOrigin']];
				}

				$dao                   = $this->productsService->fillDao($tmp);
				$dao->featureValuesIds = $featureValuesIds[$dao->getId()] ?? [];

				foreach ($dao->inSites as $v) {
					if (isset($rootCats[$v])) {
						array_unshift($dao->categories, $rootCats[$v]);
					}
				}

				if (Config::load('product.publishedByLang') && $row[0]['productTexts'][$locale]['isPublished'] === 0) {
					$dao->isActive = false;
				} else if (isset($row[0]['isPublished']) && $row[0]['isPublished'] === 0) {
					$dao->isActive = false;
				}

				if (self::$cacheEnabled) {
					$cacheDep                = $this->productsService->cacheDep;
					$cacheDep[Cache::TAGS][] = 'product/' . $dao->getId();
					$cacheDep[Cache::EXPIRE] = '1 week';
					$this->cacheService->productCache->save('product/' . $dao->getId() . '/' . $locale, $dao, $cacheDep);
				}

				$result[$tmp['id']] = $dao;
			}
		}

		return [
			'prepared' => $result,
			'loaded'   => $loaded,
		];
	}

	/**
	 * Vrátí DAO produktů z cache nebo získá data z db a uloží do cache
	 *
	 * @param int[]|string[] $ids
	 *
	 * @return Dao\Product[]
	 * @throws Throwable
	 */
	public function getProducts(array $ids, ?string $siteIdent = null): array
	{
		$baseIds = $ids;
		$data    = $this->loadBase($ids);

		/** @var Dao\Product[] $loaded */
		$loaded = $data['loaded'];

		/** @var Dao\Product[] $result */
		$result = $data['prepared'];

		if (!empty($result)) {
			$variants         = [];
			$variantParentIds = [];
			$variantIds       = [];

			if (Config::load('groupVariantsInList', false)) {
				$this->productsService->loadFresh(
					$result,
					$this->userStorage->getIdentity() ? $this->userStorage->getIdentity()->getId() : null
				);

				$this->cFreashLoaded = array_merge($this->cFreashLoaded, array_keys($result));
			}

			foreach ($result as $product) {
				if ($product->variantId) {
					$variantIds[$product->getId()] = $product->variantId;
				}

				if ($product->packageId) {
					$package = $this->packages->getPackages([$product->packageId])[$product->packageId] ?? null;

					if ($package) {
						$product->package = $package;

						$packageProductIds = array_map(static fn($v) => $v->productId, $package->items);
						$packageProducts   = $this->getProducts($packageProductIds);
						foreach ($package->items as $item) {
							if ($packageProducts[$item->productId]) {
								$item->product = $packageProducts[$item->productId];
							}
						}

						$this->packages->setMaximumQuantity($product);

						if ($product->quantity <= 0) {
							$product->setAvailability($this->availabilityService->getByIdent(Availability::SOLD_OUT));
						}
					}
				}
			}

			if (!empty($variantIds)) {
				$tmp         = [];
				$resultCache = Config::load('productsList.enableResultCache');
				$query       = $this->productsService->em->getRepository(ProductVariant::class)->createQueryBuilder('pv')
					->select('IDENTITY(pv.product) as product, pv.variantId, pv.isDefault')
					->where('pv.variantId IN (:ids)')->setParameter('ids', array_unique($variantIds))
					->getQuery();

				if ($resultCache) {
					$query->enableResultCache($resultCache);
				}

				foreach ($query->getArrayResult() as $row) {
					if ($row['isDefault'] == 1) {
						$variantParentIds[$row['variantId']] = $row['product'];
					}
					if (isset($variantIds[$row['variantId']]) && !is_array($variantIds[$row['variantId']])) {
						$variantIds[$row['variantId']] = [$variantIds[$row['variantId']]];
					}
					$tmp[$row['variantId']][] = $row['product'];
					$variants[]               = $row['product'];
				}
				$variantIds = $tmp;

				if (!empty($variants)) {
					$tmp = $this->loadBase(array_diff(array_unique($variants), $ids));

					if (Config::load('groupVariantsInList', false)) {
						$this->productsService->loadFresh(
							$tmp['prepared'],
							$this->userStorage->getIdentity() ? $this->userStorage->getIdentity()->getId() : null,
							true
						);

						$this->cFreashLoaded = array_merge($this->cFreashLoaded, array_keys($tmp['prepared']));
					}

					$loaded += $tmp['loaded'];
					$result += $tmp['prepared'];
				}
			}

			foreach ($result as $product) {
				if (isset($variantParentIds[$product->variantId])) {
					$product->variantOf = (int) $variantParentIds[$product->variantId];
				}
			}

			if (self::$loadFeatures) {
				$this->productsService->loadFeatures($result);
			}
			if (self::$loadTags) {
				$this->tagsService->loadTagsToProduct($result);
			}

			if (!Config::load('groupVariantsInList', false)) {
				$this->productsService->loadFresh(
					$result,
					$this->userStorage->getIdentity() ? $this->userStorage->getIdentity()->getId() : null,
					true
				);

				$this->cFreashLoaded = array_merge($this->cFreashLoaded, array_keys($result));
			}

			foreach ($result as $product) {
				uasort($product->features, static fn($a, $b) => $a->position <=> $b->position);
			}

			$galIds = [];
			foreach ($result as $product) {
				if ($product->galleryId) {
					$galIds[] = $product->galleryId;
				}
			}
			$galleries = $this->albumsService->getAlbums($galIds);

			for ($i = 0; $i <= 1; $i++) {
				foreach ($result as $id => $product) {
					if (!$product->isActive) {
						unset($result[$id]);
						continue;
					}

					if ($i === 0 && $product->variantOf && $product->variantOf !== $product->getId()) {
						continue;
					} else if ($i === 1 && !$product->variantOf) {
						continue;
					}

					if ($product->galleryId) {
						/** @var Album|null $gallery */
						$gallery = isset($galleries[$product->galleryId]) ? clone $galleries[$product->galleryId] : null;

						if ($gallery) {
							if ($product->helperData['variantImage']) {
								/** @var \Gallery\FrontModule\Model\Dao\Image|null $img */
								$img = $gallery->getImages()[$product->helperData['variantImage']];

								foreach ($gallery->getImages() as $image) {
									$image->isCover = $img->id == $image->id;
								}
								if ($img) {
									$gallery->setCover($img);
								}
							}

							if ($gallery->getCover()) {
								$product->variantImage = $gallery->getCover();
							}

							$product->setGallery($gallery);
						}
					}

					if ($product->manufacturerId) {
						$product->setManufacturer($this->manufacturersService->get((int) $product->manufacturerId));
					}

					if (!$product->defaultCategoryId && isset($this->cProducts[$product->variantOf])) {
						$product->defaultCategoryId = $this->cProducts[$product->variantOf]->defaultCategoryId;
					}

					if ($product->defaultCategoryId) {
						$product->defaultCategory = $this->categoriesService->get($product->defaultCategoryId);
						$product->canAddToCart    = $product->defaultCategory->canProductsAddToCart ? true : false;
					}

					if (($product->canAddToCart || !$product->defaultCategoryId) && $product->getAvailability()) {
						$product->canAddToCart = (bool) $product->getAvailability()->canAddToCart();
					}

					$this->cProducts[$id] = $product;
					$loaded[$id]          = $product;
					$result[$id]          = null;
				}
			}

			$this->eventDispatcher->dispatch(new ProductsEvent($loaded), Products::class . '::afterFillDao');

			$allLoaded          = $loaded;
			self::$allLoadedIds = array_keys($allLoaded);

			/** @phpstan-ignore-next-line */
			if (Config::load('canCacheProductsUrl') && $this->linkGenerator) {
				$linkKeys   = [];
				$whereLinks = [];
				$locale     = $this->getLocale();

				foreach ($allLoaded as $id => $product) {
					$linkKeys[] = 'link/' . $id . '/' . $locale;
				}

				foreach ($this->cacheService->productCache->bulkLoad($linkKeys) as $key => $link) {
					$tmp = explode('/', $key);
					$id  = $tmp[1];

					if ($link) {
						$allLoaded[$id]->link = $link;
					} else {
						$whereLinks[] = $id;
					}
				}

				if (!empty($whereLinks)) {
					foreach ($whereLinks as $id) {
						$product = $allLoaded[$id] ?? null;

						if (!$product) {
							continue;
						}

						$product->link = $this->generateLink($product, $locale);

						$cacheDep = [
							Cache::Tags       => 'eshopNavigation',
							Cache::EXPIRATION => '1 week',
						];
						$this->cacheService->productCache->save('link/' . $product->getId() . '/' . $locale, $product->link, $cacheDep);
					}
				}
			}

			foreach ($allLoaded as $id => $product) {

				/** @phpstan-ignore-next-line */
				if (!Config::load('canCacheProductsUrl') && $this->linkGenerator) {
					$product->link = $this->generateLink($product, $this->getLocale());
				}

				$this->cProducts[$id] = $product;
			}

			foreach (array_diff(array_keys($loaded), $baseIds) as $v) {
				unset($loaded[$v]);
			}

			if (self::$mode != self::MODE_CHECKOUT) {
				foreach ($loaded as $id => $product) {
					if (!$product->variantId) {
						continue;
					}

					foreach ($variantIds[$product->variantId] as $v) {
						if ($allLoaded[$v]) {
							$product->variants[$v] = $allLoaded[$v];
						}
					}

					if (count($product->variants) <= 1) {
						$product->variants  = [];
						$product->variantId = null;
					} else {
						foreach ($product->variants as $k => $v) {
							$product->variants[$k]->variants = &$product->variants;
						}
					}
				}
			}
		}

		$result = [];
		foreach ($ids as $id) {
			if (isset($loaded[$id])) {
				$result[$id] = $loaded[$id];
			}
		}

		$loadFresh = [];
		foreach (array_diff(array_keys($result), $this->cFreashLoaded) as $id) {
			$loadFresh[$id] = &$result[$id];
		}

		if (!empty($loadFresh)) {
			$this->productsService->loadFresh(
				$loadFresh,
				$this->userStorage->getIdentity() ? $this->userStorage->getIdentity()->getId() : null,
				true
			);
		}

		return $result;
	}

	/**
	 * @throws Throwable
	 */
	public function getProduct(int $id, ?string $siteIdent = null): ?Dao\Product
	{
		return $this->getProducts([$id], $siteIdent)[$id] ?? null;
	}

	public function loadAlternative(Dao\Product $product, int $limit): void
	{
		$data       = $this->productsService->getProductsIdInCategory($product->defaultCategoryId, 0, null, null, [], false);
		$variantIds = array_keys($product->variants);

		foreach ($data as $k => $v) {
			if ($product->getId() == $v || in_array($v, $variantIds)) {
				unset($data[$k]);
			}
		}

		shuffle($data);
		$products = $this->getProducts(array_slice($data, 0, $limit));

		$product->setAlternatives($products);
	}

	public function loadRelated(Dao\Product $product, int $limit): void
	{
		if (!Config::load('allowRelatedProducts') || !empty($product->getRelated())) {
			return;
		}

		[$related, $ids, $groupedIds] = $this->cacheService->productCache
			->load('related/' . $product->getId(), function(&$dep) use ($product) {
				$dep = [
					Cache::TAGS => ['related'],
				];

				$related    = [];
				$ids        = [];
				$groupedIds = [];
				foreach ($this->productsService->em->getRepository(RelatedProduct::class)->createQueryBuilder('rp')
					         ->select('IDENTITY(rp.product) as product, g.id as groupId, gt.name as groupName')
					         ->innerJoin('rp.group', 'g', Join::WITH, 'g.isPublished = 1')
					         ->innerJoin('g.texts', 'gt', Join::WITH, 'gt.lang = :lang')
					         ->andWhere('rp.origin = :id')
					         ->orderBy('g.position')
					         ->setParameters([
						         'lang' => $this->getLocale(),
						         'id'   => $product->getId(),
					         ])->getQuery()->getArrayResult() as $row) {
					if (!isset($related[$row['groupId']])) {
						$related[$row['groupId']] = new Dao\RelatedGroup($row['groupId'], $row['groupName']);
					}
					$ids[]                         = $row['product'];
					$groupedIds[$row['groupId']][] = $row['product'];
				}

				return [$related, $ids, $groupedIds];
			});

		if ($ids) {
			Products::$loadKey = 'related';
			$products          = $this->getProducts($ids);

			foreach ($groupedIds as $groupId => $ids) {
				shuffle($ids);
				$i = 0;

				foreach ($ids as $id) {
					if (!isset($products[$id])) {
						continue;
					}

					$related[$groupId]->addProduct($products[$id]);
					$i++;

					if ($i >= $limit) {
						break;
					}
				}
			}

			$product->setRelated($related);
			Products::$loadKey = null;
		}
	}

	public static function setMode(string $mode): void
	{
		self::$mode = $mode;
		Products::setMode(Products::MODE_CHECKOUT);
		Categories::setMode(Categories::MODE_CHECKOUT);
	}

	protected function getLocale(): string { return self::$forceLocale ?: $this->translator->getLocale(); }

	public function updateQuantitiesByIds(array $ids): void
	{
		$forUpdate = [];
		foreach ($ids as $id) {
			$product = $this->cProducts[$id] ?? null;

			if (!$product) {
				continue;
			}

			$forUpdate[$id] = &$product;
		}

		$this->productsService->loadQuantity($forUpdate, true);
	}

	protected function generateLink(Dao\Product $product, string $locale): string
	{
		$linkParams = [
			'id'     => $product->getId(),
			'locale' => $locale,
		];

		if (Config::load('useOldUrls') && $product->getExtraField(Config::load('product.oldUrlEFKey')) !== null
			&& $product->variantOf && $product->variantOf != $product->getId()) {
			$linkParams['variant'] = $product->getId();
		}

		return $this->linkGenerator->link('EshopCatalog:Front:Default:product', $linkParams);
	}

	public function clearTemp(): void
	{
		$this->cProducts = [];
		$this->cVariants = [];
	}

}
