<?php declare(strict_types = 1);

namespace EshopCatalog\Model\Entities;

use Core\Model\Entities\Country;
use Core\Model\Entities\TTranslateListener;
use Core\Model\Helpers\Traits\TExtraField;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\ORM\Mapping as ORM;
use EshopCatalog\Model\Config;
use Gallery\Model\Entities\Image;
use Gedmo\Mapping\Annotation as Gedmo;
use Gallery\Model\Entities\Album;
use Core\Model\Entities\TId;
use Nette\Utils\ArrayHash;
use Nette\Utils\DateTime;

/**
 * @ORM\Table(name="eshop_catalog__product", indexes={
 *     @ORM\Index(name="published", columns={"is_published", "id"}),
 *     @ORM\Index(name="published_cat", columns={"is_published", "id_category_default", "id"}),
 *     @ORM\Index(name="deleted_id", columns={"is_deleted", "id"}),
 *     @ORM\Index(name="disable_listing", columns={"disable_listing"}),
 * })
 * @ORM\Entity
 * @ORM\EntityListeners({"Core\Model\Entities\TranslateListener", "ProductListener"})
 */
class Product
{
	public const UNITS = ['pieces', 'kg', 'g', 'l', 'dl', 'ml'];

	use TId;
	use TTranslateListener;
	use TExtraField;

	public const EXTRA_FIELD_SECTION = 'eshopCatalogProduct';

	/**
	 * @ORM\Column(type="smallint", length=1, options={"default": 0})
	 */
	public int $isPublished = 0;

	/**
	 * @var int|bool
	 * @ORM\Column(type="smallint", length=1)
	 */
	public $inStock;

	/**
	 * @ORM\Column(type="integer", options={"default": 0}, nullable=false)
	 */
	public int $quantity;

	/**
	 * @ORM\Column(type="integer", options={"default": 1, "unsigned": true, "comment": "Minimum quantity of products per order"}, nullable=false)
	 */
	public int $minimumAmount = 1;

	/**
	 * @ORM\Column(type="integer", options={"unsigned": true, "comment": "Maximum quantity of products per order"}, nullable=true)
	 */
	public ?int $maximumAmount = null;

	/**
	 * @var double|null
	 * @ORM\Column(type="decimal", precision=10, scale=2, nullable=true)
	 */
	public $price;

	/**
	 * @var double|null
	 * @ORM\Column(type="decimal", precision=10, scale=2, nullable=true)
	 */
	public $retailPrice;

	/**
	 * @var double|null
	 * @ORM\Column(type="decimal", precision=10, scale=2, nullable=true)
	 */
	public $purchasePrice;

	/**
	 * @var Collection<string, ProductPrice>
	 * @ORM\OneToMany(targetEntity="ProductPrice", mappedBy="product", indexBy="country", cascade={"persist", "remove"})
	 */
	public Collection $prices;

	/**
	 * @var Collection<ProductPriceLevelCountry>
	 * @ORM\OneToMany(targetEntity="ProductPriceLevelCountry", mappedBy="product", cascade={"persist", "remove"})
	 */
	public Collection $priceLevelCountries;

	/**
	 * @var double|null
	 * @ORM\Column(type="decimal", precision=10, scale=2, nullable=true)
	 */
	public $recyclingFee;

	/**
	 * @ORM\ManyToOne(targetEntity="Manufacturer", cascade={"persist"})
	 * @ORM\JoinColumn(name="id_manufacturer", referencedColumnName="id", onDelete="SET NULL", nullable=true)
	 */
	protected ?Manufacturer $manufacturer = null;

	/**
	 * @var Collection<string, ProductTexts>
	 * @ORM\OneToMany(targetEntity="ProductTexts", mappedBy="id", indexBy="lang", cascade={"all"})
	 */
	protected Collection $productTexts;

	/**
	 * @var Collection<CategoryProduct>
	 * @ORM\OneToMany(targetEntity="CategoryProduct", mappedBy="product", indexBy="id_category", cascade={"all"})
	 */
	public Collection $categoryProducts;

	/**
	 * @ORM\ManyToOne(targetEntity="Category", cascade={"persist"})
	 * @ORM\JoinColumn(name="id_category_default", referencedColumnName="id", onDelete="SET NULL")
	 */
	protected ?Category $idCategoryDefault = null;

	/**
	 * @var Collection<FeatureProduct>
	 * @ORM\OneToMany(targetEntity="FeatureProduct", mappedBy="product", indexBy="featureValue", cascade={"all"})
	 */
	protected Collection $featureProducts;

	/**
	 * @var Collection<DynamicFeatureProduct>
	 * @ORM\OneToMany(targetEntity="DynamicFeatureProduct", mappedBy="product", indexBy="id", cascade={"all"})
	 */
	public Collection $dynamicFeatures;

	/**
	 * @var Collection<ProductTag>
	 * @ORM\OneToMany(targetEntity="ProductTag", mappedBy="product", indexBy="id_tag", cascade={"all"})
	 */
	public Collection $productTags;

	/**
	 * @var Collection<ProductSupplier>
	 * @ORM\OneToMany(targetEntity="ProductSupplier", mappedBy="product", indexBy="id_supplier", cascade={"all"})
	 */
	public Collection $suppliers;

	/**
	 * @ORM\ManyToOne(targetEntity="VatRate")
	 * @ORM\JoinColumn(name="id_vat_rate", referencedColumnName="id", onDelete="SET NULL")
	 */
	public ?VatRate $vatRate = null;

	/**
	 * @ORM\ManyToOne(targetEntity="Availability")
	 * @ORM\JoinColumn(name="id_availability", referencedColumnName="id", onDelete="SET NULL")
	 */
	protected ?Availability $availability = null;

	/**
	 * @ORM\ManyToOne(targetEntity="Availability")
	 * @ORM\JoinColumn(name="id_availability_after_sold_out", referencedColumnName="id", onDelete="SET NULL", nullable=true)
	 */
	public ?Availability $availabilityAfterSoldOut = null;

	/**
	 * @ORM\Column(type="string", length=20, nullable=true)
	 */
	public ?string $ean = null;

	/**
	 * @ORM\Column(name="code1", type="string", length=60, nullable=true)
	 */
	public ?string $code1 = null;

	/**
	 * @ORM\Column(name="code2", type="string", length=60, nullable=true)
	 */
	public ?string $code2 = null;

	/**
	 * @ORM\ManyToOne(targetEntity="Gallery\Model\Entities\Album", cascade={"persist"})
	 * @ORM\JoinColumn(name="gallery_id", referencedColumnName="id", onDelete="SET NULL", nullable=true)
	 */
	protected ?Album $gallery = null;

	/**
	 * @var DateTime
	 * @ORM\Column(type="datetime")
	 */
	private $created;

	/**
	 * @var DateTime|null
	 * @Gedmo\Timestampable(on="update")
	 * @ORM\Column(type="datetime", nullable=true)
	 */
	private $modified;

	/**
	 * @var Collection<ProductPriceLevel>
	 * @ORM\OneToMany(targetEntity="ProductPriceLevel", mappedBy="productId", indexBy="group_id")
	 */
	public Collection $priceLevels;

	/**
	 * @var ArrayHash|array|null
	 * @ORM\Column(type="array", nullable=true)
	 */
	protected $moreData = [];

	/**
	 * @var int|bool
	 * @ORM\Column(type="smallint", options={"default": 0}, nullable=false)
	 */
	public $unlimitedQuantity;

	/**
	 * @var int|bool
	 * @ORM\Column(type="smallint", options={"default": 0}, nullable=false)
	 */
	public $discountDisabled;

	/**
	 * @var int|bool
	 * @ORM\Column(type="smallint", options={"default": 1}, nullable=false)
	 */
	protected $categoryGiftsAllowed;

	/**
	 * @var Collection<string, ProductInSite>
	 * @ORM\OneToMany(targetEntity="ProductInSite", mappedBy="product", indexBy="site")
	 */
	public Collection $sites;

	/**
	 * @var Collection<string, ProductVariant>|null
	 */
	public ?Collection $variants = null;

	/**
	 * @var Collection<string, ProductVariant>
	 * @ORM\OneToMany(targetEntity="ProductVariant", mappedBy="product", cascade={"persist"})
	 */
	public Collection $isVariant;

	/**
	 * @var Collection<ProductDocument>
	 * @ORM\OneToMany(targetEntity="ProductDocument", mappedBy="product", cascade={"all"})
	 */
	public Collection $documents;

	/**
	 * @var Collection<ProductVideo>
	 * @ORM\OneToMany(targetEntity="ProductVideo", mappedBy="product", cascade={"all"})
	 */
	public Collection $videos;

	/**
	 * @var Collection<RelatedProduct>
	 * @ORM\OneToMany(targetEntity="RelatedProduct", mappedBy="origin", indexBy="product_id")
	 */
	public Collection $relatedProducts;

	/**
	 * @ORM\Column(type="smallint", length=1, options={"default": 0})
	 */
	public int $isAssort = 0;

	/**
	 * @ORM\Column(name="`condition`", type="string", length=255, nullable=true, options={"default": "new"})
	 */
	public ?string $condition = 'new';

	/**
	 * @ORM\Column(type="smallint", nullable=false, options={"default": 0, "unsigned": true})
	 */
	public int $disablePickUpSpedition = 0;

	/**
	 * @ORM\Column(type="smallint", nullable=false, options={"default": 0, "unsigned": true})
	 */
	public int $isOversize = 0;

	/**
	 * @ORM\Column(type="smallint", nullable=false, options={"default": 0, "unsigned": true})
	 */
	public int $disableStackInCart = 0;

	/**
	 * @ORM\Column(type="smallint", nullable=false, options={"default": 0, "unsigned": true})
	 */
	public int $isDiscount = 0;

	/**
	 * @var Collection<ProductSpedition>
	 * @ORM\OneToMany(targetEntity="ProductSpedition", mappedBy="product", indexBy="spedition_id")
	 */
	public Collection $speditions;

	/**
	 * In mm
	 * @ORM\Column(type="integer", options={"unsigned": true}, nullable=true)
	 */
	public ?int $width = null;

	/**
	 * In mm
	 * @ORM\Column(type="integer", options={"unsigned": true}, nullable=true)
	 */
	public ?int $height = null;

	/**
	 * In mm
	 * @ORM\Column(type="integer", options={"unsigned": true}, nullable=true)
	 */
	public ?int $depth = null;

	/**
	 * In g
	 * @ORM\Column(type="integer", options={"unsigned": true}, nullable=true)
	 */
	public ?int $weight = null;

	/**
	 * @ORM\Column(type="smallint", nullable=true)
	 */
	public ?int $position = null;

	/**
	 * @ORM\ManyToOne(targetEntity="Core\Model\Entities\Country")
	 * @ORM\JoinColumn(name="country_of_origin", referencedColumnName="id", onDelete="SET NULL", nullable=true)
	 */
	public ?Country $countryOfOrigin = null;

	/**
	 * @ORM\Column(type="integer", length=10, options={"unsigned": true}, nullable=true)
	 */
	public ?int $hsCustomsCode = null;

	/**
	 * @ORM\Column(type="integer", options={"unsgined": true, "default": 0}, nullable=false)
	 */
	public int $isDeleted = 0;

	/**
	 * @ORM\Column(type="string", length=255, options={"default": "pieces"}, nullable=false)
	 */
	public string $units = 'pieces';

	/**
	 * @ORM\Column(type="smallint", options={"unsgined": true, "default": 0}, nullable=false)
	 */
	public int $disableListing = 0;

	/**
	 * @ORM\ManyToOne(targetEntity="Package")
	 * @ORM\JoinColumn(name="package_id", referencedColumnName="id", nullable=true, onDelete="SET NULL")
	 */
	public ?Package $package = null;

	public function __construct()
	{
		$this->productTexts         = new ArrayCollection;
		$this->categoryProducts     = new ArrayCollection;
		$this->featureProducts      = new ArrayCollection;
		$this->suppliers            = new ArrayCollection;
		$this->productTags          = new ArrayCollection;
		$this->quantity             = 0;
		$this->inStock              = 0;
		$this->moreData             = [];
		$this->created              = new DateTime('now');
		$this->priceLevels          = new ArrayCollection;
		$this->extraFields          = new ArrayCollection;
		$this->unlimitedQuantity    = 0;
		$this->discountDisabled     = 0;
		$this->categoryGiftsAllowed = 1;
		$this->isPublished          = 1;
		$this->isVariant            = new ArrayCollection;
		$this->sites                = new ArrayCollection;
		$this->variants             = new ArrayCollection;
		$this->documents            = new ArrayCollection;
		$this->relatedProducts      = new ArrayCollection;
		$this->speditions           = new ArrayCollection;
		$this->prices               = new ArrayCollection;
		$this->priceLevelCountries  = new ArrayCollection;
		$this->dynamicFeatures      = new ArrayCollection;
		$this->position             = null;
	}

	public function __clone()
	{
		$this->id                  = null;
		$this->quantity            = 0;
		$this->gallery             = null;
		$this->isVariant           = new ArrayCollection;
		$this->productTexts        = new ArrayCollection;
		$this->categoryProducts    = new ArrayCollection;
		$this->featureProducts     = new ArrayCollection;
		$this->dynamicFeatures     = new ArrayCollection;
		$this->suppliers           = new ArrayCollection;
		$this->productTags         = new ArrayCollection;
		$this->priceLevels         = new ArrayCollection;
		$this->extraFields         = new ArrayCollection;
		$this->sites               = new ArrayCollection;
		$this->variants            = new ArrayCollection;
		$this->documents           = new ArrayCollection;
		$this->relatedProducts     = new ArrayCollection;
		$this->created             = new DateTime('now');
		$this->prices              = new ArrayCollection;
		$this->priceLevelCountries = new ArrayCollection;
		$this->isDeleted           = 0;
	}

	public function clonePrices(Product $product): array
	{
		$prices = [];
		foreach ($this->prices->toArray() as $row) {
			$price          = new ProductPrice($product, $row->getCountry(), $row->currency);
			$price->price   = $row->price;
			$price->vatRate = $row->vatRate;

			if (Config::load('product.allowRetailPrice')) {
				$price->retailPrice = $row->retailPrice;
			}

			$prices[] = $price;
		}

		return $prices;
	}

	public function clonePriceLevelCounties(Product $product): array
	{
		$prices = [];
		foreach ($this->priceLevelCountries->toArray() as $row) {
			/** @var ProductPriceLevelCountry $row */
			$price        = new ProductPriceLevelCountry($product, $row->getGroup(), $row->getCountry(), $row->currency);
			$price->price = $row->price;

			$prices[] = $price;
		}

		return $prices;
	}

	public function addProductText(string $lang): void
	{
		$this->productTexts->set($lang, new ProductTexts($this, $lang));
	}

	/**
	 * @param ProductTexts[] $texts
	 */
	public function setProductTexts(array $texts): self
	{
		$this->productTexts = new ArrayCollection($texts);

		return $this;
	}

	public function setProductText(ProductTexts $productTexts): void
	{
		$this->productTexts->set($productTexts->getLang(), $productTexts);
	}

	public function getText(?string $lang = null): ?ProductTexts
	{
		return $this->productTexts->get($lang ?? $this->locale);
	}

	/**
	 * @return Collection<string, ProductTexts>
	 */
	public function getTexts(): Collection { return $this->productTexts; }

	/**
	 * @return Collection<FeatureProduct>
	 */
	public function getFeatureProducts(): Collection
	{
		return $this->featureProducts;
	}

	/**
	 * @param FeatureProduct $featureProduct
	 */
	public function addFeatureProduct($featureProduct): void
	{
		$this->featureProducts->add($featureProduct);
	}

	/**
	 * @param FeatureProduct[] $features
	 */
	public function setFeaturesProduct(array $features): void
	{
		$this->featureProducts = new ArrayCollection($features);
	}

	public function addCategoryProduct(CategoryProduct $categoryProduct): self
	{
		if (!$this->categoryProducts->contains($categoryProduct)) {
			$this->categoryProducts->add($categoryProduct);
		}

		return $this;
	}

	public function removeCategoryProduct(CategoryProduct $categoryProduct): void
	{
		$this->categoryProducts->removeElement($categoryProduct);
	}

	/**
	 * @return Collection<CategoryProduct>
	 */
	public function getCategoryProducts(): Collection
	{
		return $this->categoryProducts;
	}

	public function removeAllCategoryProducts(): void
	{
		$this->categoryProducts->clear();
	}

	/**
	 * @param CategoryProduct[] $categories
	 */
	public function setCategories(array $categories): self
	{
		$this->categoryProducts = new ArrayCollection($categories);

		return $this;
	}

	public function removeProductTag(ProductTag $tag): self
	{
		$this->productTags->removeElement($tag);

		return $this;
	}

	public function addProductTag(ProductTag $productTag): void
	{
		$productTag->setProduct($this);
		$this->productTags->add($productTag);
	}

	/**
	 * @return Collection<ProductTag>
	 */
	public function getProductTags(): Collection
	{
		return $this->productTags;
	}

	public function setManufacturer(?Manufacturer $manufacturer = null): void
	{
		$this->manufacturer = $manufacturer;
	}

	public function getManufacturer(): ?Manufacturer
	{
		return $this->manufacturer;
	}

	/**
	 * @deprecated
	 */
	public function setDefaultCategory(Category $category): void
	{
		$this->idCategoryDefault = $category;
	}

	/**
	 * @deprecated
	 */
	public function getDefaultCategory(): ?Category
	{
		return $this->idCategoryDefault;
	}

	public function setVateRate(VatRate $vatRate): void
	{
		$this->vatRate = $vatRate;
	}

	public function getVateRate(): ?VatRate
	{
		return $this->vatRate;
	}

	public function setAvailability(?Availability $availability = null): self
	{
		$this->availability = $availability;

		return $this;
	}

	public function getAvailability(): ?Availability { return $this->availability; }

	/*************************************
	 * == Gallery
	 */

	public function getGallery(): ?Album { return $this->gallery; }

	public function setGallery(Album $album): self
	{
		$this->gallery = $album;

		return $this;
	}

	public function getImage(): ?Image
	{
		$img = null;
		if ($this->gallery) {
			if ($this->isVariant() && $this->isVariant()->defaultImage) {
				$img = $this->gallery->getImages()[$this->isVariant()->defaultImage] ?? null;
			}

			if (!$img) {
				$img = $this->gallery->getCoverImage();
			}
		}

		return $img;
	}

	/*************************************
	 * == ProductPriceLevel
	 */

	/**
	 * @return Collection<ProductPriceLevel>
	 */
	public function getPriceLevels(): Collection
	{
		return $this->priceLevels;
	}

	/*************************************
	 * == Supplier
	 */

	public function getSupplier(int $id): ?ProductSupplier
	{
		return $this->suppliers->get($id);
	}

	/**
	 * @return Collection<ProductSupplier>
	 */
	public function getSuppliers(): Collection
	{
		return $this->suppliers;
	}

	public function setSupplier(ProductSupplier $ps): void
	{
		if (!$this->suppliers->containsKey($ps->getSupplier()->getId())) {
			$this->suppliers->clear();
			$this->suppliers->add($ps);
		}
	}

	/*************************************
	 * == Dates
	 */

	/**
	 * @return DateTime
	 */
	public function getCreated() { return $this->created; }

	/**
	 * @return DateTime|null
	 */
	public function getModified() { return $this->modified; }

	public function validateCreated(): void
	{
		if ($this->created->format('y-m-d') <= 0) {
			$this->created = new DateTime;
		}
	}

	/*************************************
	 * == MoreData
	 */

	/**
	 * @param mixed $default
	 *
	 * @return mixed|null
	 */
	public function getMoreDataValue(string $key, $default = null) { return $this->moreData[$key] ?? $default; }

	/**
	 * @param mixed $value
	 */
	public function setMoreDataValue(string $key, $value): self
	{
		if (!is_array($this->moreData)) {
			$this->moreData = [];
		}

		$this->moreData[$key] = $value;

		return $this;
	}

	/**
	 * @param ArrayHash|array|null $data
	 */
	public function setMoreData($data): self
	{
		$this->moreData = $data;

		return $this;
	}

	/**
	 * @param string|int $key
	 */
	public function removeMoreData($key): Product
	{
		unset($this->moreData[$key]);

		return $this;
	}

	/*************************************
	 * == Booleans
	 */

	public function isCategoryGiftsAllowed(): int { return (int) $this->categoryGiftsAllowed; }

	/**
	 * @param string|int $allowed
	 */
	public function setCategoryGiftsAllowed($allowed): self
	{
		$this->categoryGiftsAllowed = (int) $allowed;

		return $this;
	}

	public function getIsVariantParent(): bool { return $this->isVariant() && $this->isVariant()->isDefault === 1; }

	public function getVariantParent(): ?Product
	{
		if ($this->getIsVariantParent()) {
			return $this;
		}

		foreach ($this->variants->toArray() as $v) {
			if ($v->isDefault === 1) {
				return $v->getProduct();
			}
		}

		return null;
	}

	/**
	 * @return ProductDocument[]
	 */
	public function getDocuments(): array
	{
		return $this->documents->toArray();
	}

	public function addDocument(ProductDocument $document): void
	{
		$this->documents->add($document);
	}

	public function addVideo(ProductVideo $video): void
	{
		$this->videos->add($video);
	}

	/**
	 * @return ProductVideo[]
	 */
	public function getVideos(): array
	{
		return $this->videos->toArray();
	}

	/**
	 * @return Collection<RelatedProduct>
	 */
	public function getRelatedProducts(): Collection { return $this->relatedProducts; }

	public function addRelatedProduct(RelatedProduct $relatedProduct): void
	{
		$contains = false;
		foreach ($this->relatedProducts->getKeys() as $keys) {
			if ($keys === $relatedProduct->getProduct()->getId()) {
				$contains = true;
				break;
			}
		}

		if ($this->getId() !== $relatedProduct->getProduct()->getId() && !$contains) {
			$this->relatedProducts->set($relatedProduct->getProduct()->getId(), $relatedProduct);
		}
	}

	public function isVariant(): ?ProductVariant
	{
		return $this->isVariant->first() ?: null;
	}

	/**
	 * @return Collection<ProductSpedition>
	 */
	public function getSpeditions(): Collection { return $this->speditions; }

	/**
	 * @return ProductPriceLevelCountry[][]
	 */
	public function getPriceLevelCountriesIndexedByGroup(): array
	{
		$result = [];

		foreach ($this->priceLevelCountries->toArray() as $row) {
			/** @var ProductPriceLevelCountry $row */
			$result[$row->getGroup()->getId()][$row->getCountry()->getId()] = $row;
		}

		return $result;
	}

}
