<?php declare(strict_types = 1);

namespace EshopCatalog\Model\Entities;

use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\ORM\Mapping as ORM;
use Core\Model\Entities\TId;

/**
 * @ORM\Entity
 * @ORM\Table(name="eshop_catalog__seller")
 */
class Seller
{
	use TId;

	public const EMAIL_TYPE_SELLER = 'seller';
	public const EMAIL_TYPE_SITE   = 'site';

	/**
	 * @ORM\Column(type="string", nullable=false)
	 */
	public string $title;

	/**
	 * @ORM\Column(type="string", nullable=false)
	 */
	public string $name;

	/**
	 * @ORM\Column(type="string", nullable=true)
	 */
	public ?string $address = null;

	/**
	 * @ORM\Column(type="string", nullable=true)
	 */
	public ?string $city = null;

	/**
	 * @ORM\Column(type="string", nullable=true)
	 */
	public ?string $postal = null;

	/**
	 * @ORM\Column(type="string", nullable=true)
	 */
	public ?string $country = null;

	/**
	 * @ORM\Column(name="address2", type="string", nullable=true)
	 */
	public ?string $address2 = null;

	/**
	 * @ORM\Column(name="city2", type="string", nullable=true)
	 */
	public ?string $city2 = null;

	/**
	 * @ORM\Column(name="postal2", type="string", nullable=true)
	 */
	public ?string $postal2 = null;

	/**
	 * @ORM\Column(name="country2", type="string", nullable=true)
	 */
	public ?string $country2 = null;

	/**
	 * @ORM\Column(type="string", length=12, nullable=true)
	 */
	public ?string $ic = null;

	/**
	 * @ORM\Column(type="string", length=14, nullable=true)
	 */
	public ?string $dic = null;

	/**
	 * @ORM\Column(type="string", nullable=true)
	 */
	public ?string $email = null;

	/**
	 * @ORM\Column(type="string", nullable=true)
	 */
	public ?string $phone = null;

	/**
	 * @var Collection<string, SellerInSite>
	 * @ORM\OneToMany(targetEntity="SellerInSite", mappedBy="seller", indexBy="site")
	 */
	protected Collection $sites;

	/**
	 * @var Collection<string, SellerBankAccount>
	 * @ORM\OneToMany(targetEntity="SellerBankAccount", mappedBy="seller", indexBy="currencyCode")
	 */
	protected Collection $bankAccounts;

	/**
	 * @ORM\Column(type="smallint", length=1, options={"default": 1})
	 */
	public int $sendOrder = 1;

	/**
	 * @ORM\Column(type="smallint", length=1, options={"default": 1})
	 */
	public int $sendStatus = 1;

	/**
	 * @ORM\Column(type="string", nullable=false, options={"default": "seller"})
	 */
	public string $sendEmailType = 'seller';

	public function __construct(string $title, string $name)
	{
		$this->title        = $title;
		$this->name         = $name;
		$this->sites        = new ArrayCollection;
		$this->bankAccounts = new ArrayCollection;
	}

	public function getIc(): ?string { return $this->ic; }

	public function setIc(?string $ic): self
	{
		$this->ic = $ic;

		return $this;
	}

	public function getDic(): ?string { return $this->dic; }

	public function setDic(?string $dic): self
	{
		$this->dic = $dic;

		return $this;
	}

	public function isVatPayer(): bool { return $this->dic ? true : false; }

	/**
	 * @return Collection<string, SellerInSite>
	 */
	public function getSites(): Collection { return $this->sites; }

	/**
	 * @param SellerInSite[] $sites
	 */
	public function setSites(array $sites): self
	{
		$this->sites = new ArrayCollection($sites);

		return $this;
	}

	public function getSitesIdent(): array
	{
		$sites = [];

		foreach ($this->getSites()->toArray() as $k => $v) {
			$sites[$k] = $v->getSite()->getIdent();
		}

		return $sites;
	}

	/**
	 * @return Collection<string, SellerBankAccount>
	 */
	public function getBankAccounts(): Collection { return $this->bankAccounts; }

	public function getBankAccount(?string $currencyCode = null): ?SellerBankAccount
	{
		if ($currencyCode) {
			$currencyCode = strtolower($currencyCode);
			$arr          = $this->getBankAccounts()->toArray();

			if (isset($arr[$currencyCode])) {
				return $arr[$currencyCode];
			}
		}

		return $this->bankAccounts->first() ?: null;
	}

}
