<?php declare(strict_types = 1);

namespace EshopCatalog\AdminModule\Components\Products;

use Core\Model\UI\Form\BaseContainer;
use EshopCatalog\AdminModule\Model\CategoryProducts;
use EshopCatalog\Model\Entities\CategoryProduct;
use EshopCatalog\Model\Entities\Product;
use Kdyby\Doctrine\EntityManager;
use Nette\Localization\ITranslator;
use Nette\Utils\ArrayHash;
use EshopCatalog\AdminModule\Model\Categories;
use EshopCatalog\Model\Entities\Category;

/**
 * Class CategoryContainer
 * @package EshopCatalog\AdminModule\Components\Products
 */
class CategoryContainer
{
	/** @var EntityManager */
	protected $em;

	/** @var ITranslator */
	protected $translator;

	/** @var Categories */
	protected $categoryServices;

	/** @var CategoryProducts */
	protected $categoryProductService;

	public function __construct(EntityManager $em, Categories $categories, CategoryProducts $categoryProducts, ITranslator $translator)
	{
		$this->em                     = $em;
		$this->categoryServices       = $categories;
		$this->translator             = $translator;
		$this->categoryProductService = $categoryProducts;
	}

	public function getContainer()
	{
		$container = new BaseContainer();

		$categories = [];
		$aktLevel   = 1;
		$id         = null;
		foreach ($this->categoryServices->em->getRepository(Category::class)->createQueryBuilder('c')
			         ->select('c.id, ct.name, c.lvl, c.lft, c.isPublished')
			         ->leftJoin('c.categoryTexts', 'ct', 'WITH', 'ct.lang = :lang')
			         ->setParameter('lang', $this->translator->getLocale())
			         ->andWhere('c.lvl > 0')
			         ->groupBy('c.id')
			         ->orderBy('c.root')->addOrderBy('c.lft')->getQuery()->getResult() as $c) {
			if (!$c['name'])
				continue;

			$name = (!$c['isPublished'] ? '[x ' . $c['name'] . ' x]' : $c['name']);
			$t    = '-';
			$p    = 0;
			if ($aktLevel < $c['lvl']) {
				$t = 'N';
				$p = 1;
			}
			if ($aktLevel > $c['lvl']) {
				$t = 'D';
				$p = $aktLevel - $c['lvl'];
			}
			$temp                 = $t . '_' . $p . '_' . $c['lvl'] . '_';
			$aktLevel             = $c['lvl'];
			$categories[$c['id']] = $name;

			if ($id != null) {
				$categories[$id] = $temp . $categories[$id];
			}
			$id = $c['id'];
		}
		$categories[$id] = 'D_' . $aktLevel . '_' . $categories[$id];

		$container->addRadioList('defaultCategory', 'eshopCatalog.productForm.defaultCategory', $categories);
		foreach ($categories as $key => $c) {
			$container->addCheckbox('category' . $key, $c);
		}


		return $container;
	}

	/**
	 * @param array|ArrayHash $values
	 *
	 * @return array
	 */
	public function getFormData($values): array
	{
		return (array) $values;
	}

	/**
	 * @param BaseContainer $container
	 * @param array         $values
	 */
	public function setDefaults(&$container, $values)
	{
		foreach ($container->getControls() as $c) {
			if (isset($values[$c->getName()])) {
				$c->setDefaultValue($values[$c->getName()]);
			}
		}
	}

	public function saveData(array $productsId, array $formCategories)
	{
		// Odstranit všechny kategorie
		$this->em->createQueryBuilder()->delete(CategoryProduct::class, 'cp')
			->where('cp.product IN (:products)')
			->setParameter('products', $productsId)
			->getQuery()->execute();

		// Zpracovat nové kategorie
		$defaultCategory = $formCategories['defaultCategory'] ?? null;
		$categories      = [];

		unset($formCategories['defaultCategory']);

		foreach ($formCategories as $k => $v) {
			if ($v === true) {
				$categories[] = (int) substr($k, 8);
			}
		}

		// Nastavit nové kategorie
		foreach ($productsId as $p) {
			$p = $this->em->getRepository(Product::class)->getReference($p);

			/** @var Product $p */
			if ($defaultCategory)
				$p->setDefaultCategory($this->categoryServices->getReference((int) $defaultCategory));

			$tmp = [];
			foreach ($categories as $c)
				$tmp[] = new CategoryProduct($p, $this->categoryServices->getReference($c));
			$this->em->persist($tmp);
		}
	}
}
