<?php declare(strict_types = 1);

namespace EshopCatalog\FrontModule\Model;

use Core\Model\Helpers\BaseFrontEntityService;
use EshopCatalog\Model\Entities\Manufacturer;
use Nette\Caching\Cache;
use EshopCatalog\FrontModule\Model\Dao;

/**
 * Class Manufacturers
 * @package EshopCatalog\FrontModule\Model
 */
class Manufacturers extends BaseFrontEntityService
{

	const CACHE_NAMESPACE = 'manufacturers';

	protected $entityClass = Manufacturer::class;

	/** @var CacheService */
	protected $cacheService;

	/** @var array */
	protected $cManufacturers;

	/** @var array */
	protected $cacheDep = [
		Cache::TAGS   => ['manufacturers'],
		Cache::EXPIRE => '1 week',
	];

	/**
	 * Manufacturers constructor.
	 *
	 * @param CacheService $cacheService
	 */
	public function __construct(CacheService $cacheService)
	{
		$this->cacheService = $cacheService;
	}

	public function getCache()
	{
		if ($this->cache === null)
			$this->cache = new Cache($this->cacheStorage, self::CACHE_NAMESPACE);

		return $this->cache;
	}

	/**
	 * @return Dao\Manufacturer[]
	 */
	public function getManufacturers(): array
	{
		if ($this->cManufacturers === null) {
			$this->cManufacturers = [];

			$locale = $this->translator->getLocale();
			$key    = 'manufacturers/' . $locale;

			$this->cManufacturers = $this->cacheService->defaultCache->load($key, function(&$dep) use ($locale) {
				$dep    = $this->cacheDep;
				$result = [];

				foreach ($this->getEr()->createQueryBuilder('m')
					         ->addSelect('mt')
					         ->innerJoin('m.texts', 'mt', 'WITH', 'mt.lang = :lang')
					         ->andWhere('m.isPublished = 1')
					         ->setParameter('lang', $locale)
					         ->getQuery()->getArrayResult() as $row) {
					$result[$row['id']] = $this->fillDao($row);
				}

				return $result;
			});
		}

		return $this->cManufacturers;
	}

	/**
	 * @param int $id
	 *
	 * @return Dao\Manufacturer|null
	 */
	public function get(int $id): ?Dao\Manufacturer
	{
		return $this->getManufacturers()[$id] ?? null;
	}

	/**
	 * @param array $a
	 *
	 * @return Dao\Manufacturer[]
	 */
	protected function fillDao($a)
	{
		$locale = $this->translator->getLocale();

		$manufacturer = (new Dao\Manufacturer())
			->setId($a['id'])
			->setName($a['name'])
			->setLogo($a['logo'])
			->setDescription($a['texts'][$locale]['description']);

		return $manufacturer;
	}
}
