<?php declare(strict_types = 1);

namespace EshopCatalog\AdminModule\Components\Features;

use Core\Model\Event\CreateFormEvent;
use Core\Model\Event\FormSuccessEvent;
use Core\Model\Event\SetFormDataEvent;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use EshopCatalog\Model\Config;
use Nette\InvalidArgumentException;
use Nette\Utils\ArrayHash;
use EshopCatalog\Model\Entities\Feature;
use EshopCatalog\Model\Entities\FeatureTexts;
use EshopCatalog\AdminModule\Model\Features;

class FeatureForm extends BaseControl
{
	public ?Feature $feature = null;

	public ?FeatureTexts $featureTexts = null;

	protected Features $featureServices;

	public function __construct(Features $features)
	{
		$this->featureServices = $features;
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	protected function createComponentForm(): BaseForm
	{
		$form = $this->createForm();

		$form->addText('name', 'eshopCatalog.featureForm.name')
			->setIsMultilanguage()
			->setMaxLength(255);
		$form->addBool('isPublished', 'eshopCatalog.featureForm.isPublished')->setDefaultValue(1);
		$form->addBool('useAsFilter', 'eshopCatalog.featureForm.useAsFilter')->setDefaultValue(1);
		$form->addBool('exportHeureka', 'eshopCatalog.featureForm.exportHeureka')->setDefaultValue(1);
		$form->addBool('exportZbozi', 'eshopCatalog.featureForm.exportZbozi')->setDefaultValue(1);

		if (Config::load('features.allowDescription')) {
			$form->addText('productTabTitle', 'eshopCatalog.featureForm.productTabTitle')
				->setIsMultilanguage()
				->setMaxLength(255);
		}

		$this->eventDispatcher->dispatch(
			new CreateFormEvent($form, $this->getPresenter(false) ? $this->template : null),
			FeatureForm::class . '::createForm'
		);

		$form->addSaveCancelControl('saveControl');

		$form->onSuccess[] = [$this, 'formSuccess'];

		return $form;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values): bool
	{
		try {
			$feature = $this->feature ?: new Feature();

			$feature->setIsPublished($values->isPublished);
			$feature->setUseAsFilter($values->useAsFilter);
			$feature->exportHeureka = $values->exportHeureka;
			$feature->exportZbozi   = $values->exportZbozi;

			$langValues = $form->convertMultilangValuesToArray();
			$texts      = $feature->getFeatureTexts()->toArray();

			foreach ($langValues as $lang => $vals) {
				if ($vals['name'] == '') {
					if (isset($texts[$lang])) {
						$this->em->remove($texts[$lang]);
						unset($texts[$lang]);
					}
					continue;
				}

				$text       = $texts[$lang] ?? new FeatureTexts($feature, $lang);
				$text->name = $vals['name'];

				if (Config::load('features.allowDescription'))
					$text->productTabTitle = $vals['productTabTitle'];

				$this->em->persist($text);
				$texts[$lang] = $text;
			}

			$feature->setFeatureTexts($texts);

			$event                   = new FormSuccessEvent($form, $values,
				$this->getPresenter(false) ? $this->template : null, $this->getPresenter(false) ? $this->getPresenter() : null);
			$event->custom['entity'] = $feature;
			$this->eventDispatcher->dispatch($event, FeatureForm::class . '::formSuccess');

			$this->em->persist($feature);
			$this->em->flush();
			$form->addCustomData('featureId', $feature->getId());
			$this->getPresenter()->flashMessageSuccess('eshopCatalog.featureForm.featureSaved');

			return true;
		} catch (\Exception $e) {
			$form->addError($e->getMessage());
		}

		return false;
	}

	public function setFeature($id)
	{
		$this->feature = $this->featureServices->get($id);

		if ($this->feature) {
			$form = $this['form'];
			$d    = [
				'isPublished'   => $this->feature->getIsPublished(),
				'useAsFilter'   => $this->feature->getUseAsFilter(),
				'exportHeureka' => $this->feature->exportHeureka,
				'exportZbozi'   => $this->feature->exportZbozi,
			];
			$form->setDefaults($d);

			$mTexts = $form->prepareMultilangTextsForForm([
				'name',
				'productTabTitle',
			], $this->feature->getFeatureTexts()->toArray());
			$form->setDefaults($mTexts['texts']);

			$this->eventDispatcher->dispatch(new SetFormDataEvent($form, $this->feature), FeatureForm::class . '::setFeature');
		} else
			throw new InvalidArgumentException();
	}
}

