<?php declare(strict_types = 1);

namespace EshopCatalog\AdminModule\Components\Features;

use Blog\AdminModule\Model\Articles;
use Blog\DI\BlogExtension;
use Core\Model\Event\CreateFormEvent;
use Core\Model\Event\FormSuccessEvent;
use Core\Model\Event\SetFormDataEvent;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseContainer;
use Core\Model\UI\Form\BaseForm;
use EshopCatalog\Model\Config;
use Navigations\AdminModule\Model\Navigations;
use Nette\DI\Container;
use Nette\InvalidArgumentException;
use Nette\Utils\ArrayHash;
use EshopCatalog\Model\Entities\Feature;
use EshopCatalog\Model\Entities\FeatureTexts;
use EshopCatalog\Model\Entities\FeatureValue;
use EshopCatalog\Model\Entities\FeatureValueTexts;
use EshopCatalog\AdminModule\Model\FeatureValues;

class FeatureValueForm extends BaseControl
{
	public ?featureValue $featureValue = null;

	public ?featureValueTexts $featureValueTexts = null;

	protected FeatureValues $featureValueServices;

	protected Navigations $navigations;

	protected Container $container;

	public function __construct(FeatureValues $featureValues, Navigations $navigations, Container $container)
	{
		$this->featureValueServices = $featureValues;
		$this->navigations          = $navigations;
		$this->container            = $container;
	}

	public function render(): void
	{
		$this->template->thisForm = $this['form'];
		$this->template->render($this->getTemplateFile());
	}

	protected function createComponentForm(): BaseForm
	{
		$form = $this->createForm();

		$features = [null => ''];
		foreach ($this->em->getRepository(Feature::class)->createQueryBuilder('f')->select('f.id, ft.name')
			         ->join(FeatureTexts::class, 'ft', 'WITH', 'f.id = ft.id')
			         ->where('ft.lang = :lang')->setParameter('lang', $this->translator->getLocale())
			         ->getQuery()->getResult() as $g) {
			$features[$g['id']] = $g['name'];
		}

		if (count($features) == 1) {
			return $form;
		}

		$form->addSelect('feature', 'eshopCatalog.featureValueForm.feature', $features)->setRequired();
		$form->addText('value', 'eshopCatalog.featureValueForm.value')->setIsMultilanguage()->setMaxLength(255);

		if (Config::load('features.allowImage')) {
			$form->addFilesManager('image', 'eshopCatalog.featureValueForm.image');
		}

		if (Config::load('features.allowDescription')) {
			$form->addBool('showAsTag', 'eshopCatalog.featureValueForm.showAsTag')
				->setDefaultValue(0);
			$form->addText('tagText', 'eshopCatalog.featureValueForm.tagText')
				->setIsMultilanguage()
				->setMaxLength(255);
			$form->addColorInput('tagTextColor', 'eshopCatalog.featureValueForm.tagTextColor')
				->setDefaultValue(null);
			$form->addColorInput('tagBgColor', 'eshopCatalog.featureValueForm.tagBgColor')
				->setDefaultValue(null);
			$form->addEditor('shortDescription', 'eshopCatalog.featureValueForm.shortDescription')
				->setToolbar('Text')
				->setDisableAutoP(false)
				->setHeight(100)
				->setIsMultilanguage();
			if (!Config::load('features.allowShortDescription'))
				$form['shortDescription']->getLabelPrototype()->class[] = 'disabled';
			$form->addEditor('longDescription', 'eshopCatalog.featureValueForm.longDescription')
				->setHeight(300)
				->setDisableAutoP(false)
				->setIsMultilanguage();

			$moreLink = $form->addContainer('moreLink', 'eshopCatalog.featureValueForm.moreLink');
			$moreLink->addText('text', 'eshopCatalog.featureValueForm.moreLinkText');
			$moreLink->addSelect('navigation', 'eshopCatalog.featureValueForm.moreLinkNavigation', ['' => ''] + $this->navigations->getOptionsForSelect());
			if (class_exists(BlogExtension::class) && $this->container->hasService('blog.admin.articles')) {
				/** @var Articles $articlesService */
				$articlesService = $this->container->getService('blog.admin.articles');
				$moreLink->addSelect('article', 'eshopCatalog.featureValueForm.moreLinkArticle', ['' => ''] + $articlesService->getOptionsForSelect());
			}
		}

		$this->eventDispatcher->dispatch(
			new CreateFormEvent($form, $this->getPresenter(false) ? $this->template : null),
			FeatureValueForm::class . '::createForm'
		);

		$form->addSubmit('saveAndNew', null ?: 'eshopCatalog.defaultForm.saveAndNew')->getControlPrototype()->addHtml('<i class="fa fa-save"></i>');
		$form->addSaveCancelControl('saveControl');

		$form->onSuccess[] = [$this, 'formSuccess'];

		return $form;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values): bool
	{
		try {
			if ($this->featureValue) {
				$featureValue = $this->featureValue;
			} else {
				$featureValue = new FeatureValue();
			}

			$featureValue->setFeature($this->em->getReference(Feature::class, $values->feature));

			if (Config::load('features.allowImage'))
				$featureValue->image = $values->image;

			if (Config::load('features.allowDescription')) {
				$featureValue->showAsTag    = (int) $values->showAsTag;
				$featureValue->tagTextColor = $values->tagTextColor;
				$featureValue->tagBgColor   = $values->tagBgColor;

				$moreLinkValues = $values->moreLink;
				$moreLink       = null;
				foreach (['text', 'navigation', 'article'] as $v) {
					if ($moreLinkValues->$v) {
						$moreLink = $v . '|' . $moreLinkValues->$v;
						break;
					}
				}
				$featureValue->moreLink = $moreLink;
			}

			$texts      = $featureValue->getFeatureValueTexts()->toArray();
			$langValues = $form->convertMultilangValuesToArray();

			$this->em->persist($featureValue);
			foreach ($langValues as $lang => $vals) {
				if ($vals['value'] == '') {
					if (isset($texts[$lang])) {
						$this->em->remove($texts[$lang]);
						unset($texts[$lang]);
					}
					continue;
				}

				$text = $texts[$lang] ?? new FeatureValueTexts($featureValue, $lang);
				$text->setName($vals['value']);

				if (Config::load('features.allowDescription')) {
					$text->tagText          = $vals['tagText'];
					$text->shortDescription = $vals['shortDescription'];
					$text->longDescription  = $vals['longDescription'];
				}

				$this->em->persist($text);

				$texts[$lang] = $text;
			}

			$featureValue->setTexts($texts);

			$event                   = new FormSuccessEvent($form, $values,
				$this->getPresenter(false) ? $this->template : null, $this->getPresenter(false) ? $this->getPresenter() : null);
			$event->custom['entity'] = $featureValue;
			$this->eventDispatcher->dispatch($event, FeatureValueForm::class . '::formSuccess');

			$this->em->persist($featureValue);
			$this->em->flush();
			$form->addCustomData('featureValueId', $featureValue->getId());
			$form->addCustomData('featureId', $featureValue->getFeature()->getId());
			$this->getPresenter()->flashMessageSuccess('eshopCatalog.featureValueForm.featureValueSaved');

			return true;
		} catch (\Exception $e) {
			$form->addError($e->getMessage());
		}

		return false;
	}

	public function setFeature($id): void
	{
		$this['form']->setDefaults([
			'feature' => $id,
		]);
		$this['form']->addCustomData('featureId', $id);
	}

	public function setFeatureValue($id): void
	{
		$this->featureValue = $this->em->getRepository(FeatureValue::class)->find($id);

		if (!$this->featureValue)
			throw new InvalidArgumentException();

		$form = $this['form'];
		$d    = [
			'feature'      => $this->featureValue->getFeature()->getId(),
			'showAsTag'    => $this->featureValue->showAsTag,
			'image'        => $this->featureValue->image,
			'tagTextColor' => $this->featureValue->tagTextColor,
			'tagBgColor'   => $this->featureValue->tagBgColor,
		];

		if ($this->featureValue->moreLink) {
			$v                    = explode('|', $this->featureValue->moreLink);
			$d['moreLink'][$v[0]] = $v[1];
		}

		$form->setDefaults($d);

		$mTexts = $form->prepareMultilangTextsForForm([
			'value' => 'name',
			'tagText',
			'shortDescription',
			'longDescription',
		], $this->featureValue->getFeatureValueTexts()->toArray());
		$form->setDefaults($mTexts['texts']);
		$this->eventDispatcher->dispatch(new SetFormDataEvent($form, $this->featureValue), FeatureValueForm::class . '::setFeatureValue');
	}
}

