<?php declare(strict_types = 1);

namespace EshopCatalog\AdminModule\Components\Related;

use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use EshopCatalog\AdminModule\Model\Repository\RelatedGroups;
use EshopCatalog\Model\Entities\RelatedGroup;
use EshopCatalog\Model\Entities\RelatedGroupText;
use Nette\InvalidArgumentException;
use Nette\Utils\ArrayHash;

class RelatedGroupForm extends BaseControl
{
	/** @var RelatedGroup */
	public ?RelatedGroup $relatedGroup = null;

	/** @var int @persistent */
	public ?int $id = null;

	/** @var RelatedGroups */
	protected RelatedGroups $relatedGroups;

	public function __construct(RelatedGroups $relatedGroups)
	{
		$this->relatedGroups = $relatedGroups;
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	protected function createComponentForm(): BaseForm
	{
		$form = $this->createForm();
		$form->setAjax();

		$form->addText('name', 'eshopCatalog.relatedGroup.name')
			->setIsMultilanguage()
			->setMaxLength(255);
		$form->addBool('isPublished', 'eshopCatalog.relatedGroup.isPublished')
			->setDefaultValue(1);

		$form->addSaveCancelControl('saveControl');

		$form->onSuccess[] = [$this, 'formSuccess'];

		return $form;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values): bool
	{
		try {
			$langValues = $form->convertMultilangValuesToArray();
			/** @var RelatedGroupText[] $texts */

			if ($this->id) {
				$group = $this->relatedGroups->getReference($this->id);
				$texts = $group->getTexts()->toArray();
			} else {
				$group = new RelatedGroup();
				$texts = [];
			}

			foreach ($langValues as $l => $v) {
				if ($v['name'] == '') {
					if (isset($texts[$l])) {
						$this->em->remove($texts[$l]);
						unset($texts[$l]);
					}
					continue;
				}

				if (!$texts[$l])
					$texts[$l] = new RelatedGroupText($group, $l);
				$texts[$l]->setName($v['name']);

				$this->em->persist($texts[$l]);
			}

			$group->setTexts($texts);
			$group->isPublished = $values->isPublished ? 1 : 0;

			$this->em->persist($group);
			$this->em->flush();
			$form->addCustomData('relatedGroupId', $group->getId());
			$this->getPresenter()->flashMessageSuccess('default.saved');
		} catch (\Exception $e) {
			$form->addError($e->getMessage());
			$this->redrawControl('form');

			return false;
		}

		return true;
	}

	public function setRelatedGroup(int $id): void
	{
		$this->id = $id;
		$group    = $this->relatedGroup = $this->relatedGroups->get($id);

		if (!$group)
			throw new InvalidArgumentException('Group not found');

		$this['form']->setDefaults([
			'isPublished' => $group->isPublished,
		]);

		$texts = [];
		foreach ($group->getTexts()->toArray() as $lang => $val) {
			/** @var RelatedGroupText $val */
			$texts['name'][$lang] = $val->getName();
		}
		$this['form']->setDefaults($texts);
	}
}
