<?php declare(strict_types = 1);

namespace EshopCatalog\FrontModule\Model;

use Core\Model\Entities\QueryBuilder;
use Core\Model\Helpers\Strings;
use Currency\Model\Exchange;
use Doctrine\ORM\Query\Expr\Join;
use EshopCatalog\FrontModule\Model\Dao\FilterGroup;
use EshopCatalog\FrontModule\Model\Dao\FilterItem;
use EshopCatalog\Model\Config;
use EshopCatalog\Model\Entities\CategoryFilter;
use EshopCatalog\Model\Entities\FeatureProduct;
use EshopCatalog\Model\Entities\Product;
use Core\Model\Entities\EntityManagerDecorator;
use EshopCatalog\Model\Entities\ProductTag;
use Nette\Caching\Cache;
use Nette\SmartObject;
use Nette\Utils\DateTime;

/**
 * TODO misto hledání podle productsId používat třeba kategorie
 *
 * Class FilterService
 * @package EshopCatalog\FrontModule\Model
 */
class FilterService
{
	use SmartObject;

	public static array $prepareDataCategories = [];

	const CACHE_NAMESPACE = 'eshopCatalogFilter';

	protected EntityManagerDecorator $em;

	protected CacheService $cacheService;

	protected Features $featuresService;

	protected Tags $tags;

	protected Manufacturers $manufacturers;

	/** @var Config */
	protected $config;

	protected Exchange $exchange;

	/** @var array */
	protected $cProductsData;

	/** @var array */
	public static $cacheTags = [];

	/** @var array */
	protected $cacheDep = [
		Cache::TAGS   => ['filters'],
		Cache::EXPIRE => '10 minutes',
	];

	protected ?array $cFeatureProducts = null;

	protected ?array $cProducts = null;

	/** @var array */
	protected $cFeatures, $cFeatureValues;

	public function __construct(EntityManagerDecorator $em, CacheService $cacheService, Manufacturers $manufacturers,
	                            Features $features, Config $config, Tags $tags, Exchange $exchange)
	{
		$this->em              = $em;
		$this->cacheService    = $cacheService;
		$this->manufacturers   = $manufacturers;
		$this->featuresService = $features;
		$this->config          = $config;
		$this->tags            = $tags;
		$this->exchange        = $exchange;
	}

	/**
	 * @param int[] $productIds
	 *
	 * @return FilterGroup[]
	 */
	public function getFeatures(array $productIds = [])
	{
		$groups = [];
		$qb     = $this->em->getRepository(FeatureProduct::class)->createQueryBuilder('fp')
			->select('IDENTITY(fv.feature) as featureId, IDENTITY(fp.featureValue) as valueId')
			->join('fp.featureValue', 'fv', 'WITH', 'fv.isPublished = 1')
			->join('fv.feature', 'f', 'WITH', 'f.useAsFilter = 1')
			->orderBy('f.position')->addOrderBy('fv.position');

		if ($productIds) {
			$qb->where('fp.product IN (' . implode(',', array_values($productIds)) . ')');
		}

		$data = $qb->getQuery()->getScalarResult();

		foreach ($data as $row) {
			if (!isset($groups[$row['featureId']])) {
				$tmp = $this->featuresService->getFeatureById($row['featureId']);

				if ($tmp && $tmp['name']) {
					$tmpV       = new FilterGroup();
					$tmpV->id   = $tmp['id'];
					$tmpV->name = $tmp['name'];
					$tmpV->type = 'feature';

					$groups[$row['featureId']] = $tmpV;
				}
			}

			$g = $groups[$row['featureId']];

			if (!isset($g->items[$row['valueId']])) {
				$tmp = $this->featuresService->getFeatureValueById($row['valueId']);

				if ($tmp && $tmp['name']) {
					$tmpV        = new FilterItem();
					$tmpV->id    = $tmp['id'];
					$tmpV->value = $tmp['name'];
					$tmpV->type  = 'featureValue';
					$tmpV->rodM  = $tmp['rodM'];
					$tmpV->rodZ  = $tmp['rodZ'];
					$tmpV->rodS  = $tmp['rodS'];

					$g->items[$row['valueId']] = $tmpV;
				}
			}
		}

		return $groups;
	}

	/**
	 * @param int[] $productIds
	 *
	 * @return array
	 * @throws \Doctrine\ORM\NonUniqueResultException
	 */
	public function getMinMaxPrice($productIds)
	{
		$data = [];

		$tmp = $this->em->getRepository(Product::class)->createQueryBuilder('p')
			->select('MIN(p.price), MAX(p.price)')
			->where('p.id IN (:ids)')->setParameter('ids', $productIds)
			->getQuery()->getScalarResult();
		if (isset($tmp[0])) {
			$data = [
				'min' => $this->exchange->change((float) $tmp[0][1]),
				'max' => $this->exchange->change((float) $tmp[0][2]),
			];
		}

		return $data;
	}

	/**
	 * @param $productIds
	 *
	 * @return FilterItem[]
	 */
	public function getManufacturers($productIds): array
	{
		$toUppercase = $this->config->get('filterManufacturersUppercase', false);
		$data        = [];

		$prods = array_intersect_key($this->manufacturers->getManufacturers(), $this->getProductsData($productIds)['m'] ?? []);
		uasort($prods, fn($a, $b) => $a->name <=> $b->name);

		foreach ($prods as $m) {
			$data[$m->id]        = new FilterItem();
			$data[$m->id]->id    = $m->id;
			$data[$m->id]->value = $toUppercase ? mb_strtoupper($m->name) : $m->name;
		}

		return $data;
	}

	/**
	 * @param $productIds
	 *
	 * @return FilterItem[]
	 */
	public function getTags($productIds): array
	{
		$data = [];

		foreach (array_intersect_key($this->tags->getAll(), $this->getProductsData($productIds)['t'] ?? []) as $tag) {
			$data[$tag->id]        = new FilterItem();
			$data[$tag->id]->id    = $tag->id;
			$data[$tag->id]->value = $tag->name;

			$data[$tag->id]->customData['tagColor'] = [
				'text' => $tag->textColor,
				'bg'   => $tag->bgColor,
			];
		}

		return $data;
	}

	/**
	 * @param int  $id
	 * @param bool $deep
	 *
	 * @return array
	 */
	public function getCategoryFilters(int $id, $deep = true)
	{
		$key  = 'categoryFilters_' . $id . ($deep ? '_deep' : '');
		$data = [];

		$features = $this->cacheService->categoryCache->load($key, function(&$dep) use ($id, $deep) {
			$dep                = $this->cacheDep;
			$dep[Cache::EXPIRE] = '14 days';
			$dep[Cache::TAGS][] = 'category/' . $id;

			$find = function(int $id) {
				return $this->em->getRepository(CategoryFilter::class)->createQueryBuilder('f')
					->select('IDENTITY(f.feature) as featureId')
					->andWhere('f.category = :category')->setParameters(['category' => $id])
					->orderBy('f.position')
					->getQuery()->getArrayResult();
			};

			$features = $find($id);

			if (!$features && $deep) {
				try {
					$stmt = $this->em->getConnection()->prepare("
						SELECT T2.id 
						FROM (
							SELECT @r AS _id,
								(SELECT @r := parent_id FROM eshop_catalog__category WHERE id = _id) AS parent_id, @l := @l + 1 AS lvl 
								FROM (SELECT @r := {$id}, @l := 0) vars, eshop_catalog__category h WHERE @r <> 0) T1 
						JOIN eshop_catalog__category T2 ON T1._id = T2.id ORDER BY T1.lvl DESC
						");
					$stmt->execute();
					$tmp = $stmt->fetchAll();
				} catch (\Exception $e) {
					$tmp = [];
				}
				array_shift($tmp);
				array_pop($tmp);
				$categories = [];
				foreach ($tmp as $row) {
					$dep[Cache::TAGS][] = 'category/' . $row['id'];
					$categories[]       = $row['id'];
				}

				while (!empty($categories) && empty($features)) {
					$id       = array_pop($categories);
					$features = $find((int) $id);

					if ($features)
						break;
				}
			}

			return array_map(function($row) { return $row['featureId']; }, $features);
		});

		if (!$features)
			return $data;

		foreach ($features as $featureId) {
			$tmp = $this->featuresService->getFeatureById($featureId);

			$group       = new FilterGroup();
			$group->id   = (int) $tmp['id'];
			$group->name = $tmp['name'];
			$group->type = 'feature';

			$data[$featureId] = $group;

			foreach ($this->featuresService->getValuesByFeatureId((int) $featureId) as $tmp) {
				$item        = new FilterItem();
				$item->id    = (int) $tmp['id'];
				$item->value = $tmp['name'];
				$item->type  = 'featureValue';

				$data[$featureId]->items[$tmp['id']] = $item;
			}
		}

		return $data;
	}

	protected function prepareData(): array
	{
		$data = [];
		$ids  = !empty(self::$prepareDataCategories) ? implode(',', self::$prepareDataCategories) : null;
		$now  = (new DateTime())->format('Y-m-d H:i:00');

		// Base
		$baseQb = $this->em->createQueryBuilder()->select('p.id as id, p.price as price, IDENTITY(p.manufacturer) as m')
			->from(Product::class, 'p')
			->andWhere('p.isPublished = 1');
		foreach ($this->prepareDataExecuteQuery($baseQb, 'base', $ids) as $row) {
			$data[$row['id']]      = $row;
			$data[$row['id']]['m'] = $row['m'];
		}

		// Features
		$baseQb = $this->em->createQueryBuilder()->select('IDENTITY(fp.featureValue) as f, IDENTITY(fp.product) as id')
			->from(FeatureProduct::class, 'fp')
			->innerJoin('fp.product', 'p', Join::WITH, 'p.isPublished = 1');
		foreach ($this->prepareDataExecuteQuery($baseQb, 'features', $ids) as $row) {
			$data[$row['id']]['f'][] = $row['f'];
		}

		// Tags
		$baseQb = $this->em->createQueryBuilder()->select('IDENTITY(pt.tag) as tag, IDENTITY(pt.product) as id')
			->from(ProductTag::class, 'pt')
			->andWhere('pt.validFrom IS NULL OR pt.validFrom <= :now')
			->andWhere('pt.validTo IS NULL OR pt.validTo >= :now')
			->setParameter('now', $now)
			->innerJoin('pt.product', 'p', Join::WITH, 'p.isPublished = 1');
		foreach ($this->prepareDataExecuteQuery($baseQb, 'tags', $ids) as $row) {
			$data[$row['id']]['tag'][] = $row['tag'];
		}

		return $data;
	}

	protected function prepareDataExecuteQuery(QueryBuilder $baseQb, string $cacheKey, ?string $ids = null)
	{
		if ($ids)
			$cacheKey .= '_' . Strings::webalize($ids);

		$result = $this->cacheService->filterCache->load($cacheKey, function(&$dep) use ($baseQb, $ids) {
			$dep = $this->cacheDep;

			if ($ids) {
				$qb1 = (clone $baseQb)->innerJoin('p.sites', 's', Join::WITH, 's.category IN (' . $ids . ') AND s.isActive = 1')
					->innerJoin('s.category', 'sc', Join::WITH, 'sc.isPublished = 1');
				$qb2 = (clone $baseQb)->innerJoin('p.categoryProducts', 'cp', Join::WITH, 'cp.category IN (' . $ids . ')')
					->innerJoin('cp.category', 'cpc', Join::WITH, 'cpc.isPublished = 1');

				return array_merge(
					$qb1->getQuery()->getScalarResult(),
					$qb2->getQuery()->getScalarResult(),
				);
			} else {
				return $baseQb->getQuery()->getScalarResult();
			}
		});

		return $result;
	}

	public function getProductsData(array $productsIds, string $cacheKey = '')
	{
		if (!$this->cProductsData) {
			$result = [
				'f' => [],
				'm' => [],
				'p' => [],
				't' => [],
			];

			$data = $this->prepareData();

			foreach (array_intersect_key($data, array_flip($productsIds)) as $v) {
				$result['p'][$v['id']]['price']   = $v['price'];
				$result['p'][$v['id']]['vatRate'] = 21;
				$result['p'][$v['id']]['id']      = $v['id'];

				if ($v['f']) {
					foreach ($v['f'] as $f) {
						$result['f'][$f][]            = $v['id'];
						$result['p'][$v['id']]['f'][] = $f;
					}
				}

				if ($v['tag']) {
					foreach ($v['tag'] as $t)
						$result['t'][$t][] = $v['id'];
					$result['p'][$v['id']]['t'][] = $t;
				}

				if ($v['m']) {
					$result['m'][$v['m']][$v['id']] = $v['id'];
					$result['p'][$v['id']]['m']     = $v['m'];
				} else {
					$result['m'][0][$v['id']]   = $v['id'];
					$result['p'][$v['id']]['m'] = $v['m'];
				}
			}

			$this->cProductsData = $result;
		}

		return $this->cProductsData;
	}
}
