<?php declare(strict_types = 1);

namespace EshopCatalog\Model;

use Core\Model\Entities\EntityManagerDecorator;
use EshopCatalog\AdminModule\Model\Helpers\PricesHelper;
use EshopCatalog\Model\Entities\Product;
use EshopCatalog\Model\Entities\ProductPrice;
use EshopCatalog\Model\Entities\ProductPriceLevel;
use EshopCatalog\Model\Entities\ProductTag;
use EshopCatalog\Model\Entities\ProductTexts;
use EshopCatalog\Model\Entities\ProductVariant;
use EshopGifts\DI\EshopGiftsExtension;
use EshopGifts\Model\Entities\ProductGift;
use EshopOrders\Model\Entities\GroupCustomers;

class Products
{
	protected EntityManagerDecorator $em;

	protected PricesHelper $pricesHelper;

	protected static array $processed = [];

	public function __construct(EntityManagerDecorator $em, PricesHelper $pricesHelper)
	{
		$this->em           = $em;
		$this->pricesHelper = $pricesHelper;
	}

	public function updateVariant(ProductVariant $va, Product $parent)
	{
		$em     = $this->em;
		$vaProd = $va->getProduct();

		if (in_array($vaProd->getId(), self::$processed))
			return;

		self::$processed[] = $vaProd->getId();

		/** @var ProductTexts[] $vaProdTexts */
		$vaProdTexts = $vaProd->getTexts()->toArray();

		$va->createdDefault = $parent->getCreated();

		foreach ($parent->getTexts()->toArray() as $lang => $text) {
			if (!isset($vaProdTexts[$lang])) {
				$vaProdTexts[$lang] = new ProductTexts($vaProd, $lang);
				$vaProdTexts[$lang]->setName($text->name);
				$vaProdTexts[$lang]->setName2($text->name2);
				$vaProdTexts[$lang]->shortDescription = $text->shortDescription;
				$vaProdTexts[$lang]->description      = $text->description;
			} else {
				/** @var ProductTexts $text */
				if (!$va->useName)
					$vaProdTexts[$lang]->setName($text->name);
				if (!$va->useName2)
					$vaProdTexts[$lang]->setName2($text->name2);
				if (!$va->useShortDescription)
					$vaProdTexts[$lang]->shortDescription = $text->shortDescription;
				if (!$va->useDescription)
					$vaProdTexts[$lang]->description = $text->description;
			}

			$em->persist($vaProdTexts[$lang]);
		}

		if (!$va->usePriceLevels) {
			$vaPriceLevels = $vaProd->getPriceLevels()->toArray();
			foreach ($parent->getPriceLevels()->toArray() as $id => $v) {
				/** @var ProductPriceLevel $v */
				$g = $vaPriceLevels[$id] ?? new ProductPriceLevel($vaProd, $em->getReference(GroupCustomers::class, $id));

				$g->price = $v->price;

				$em->persist($g);
			}

			$priceLevelCountries = [];
			foreach ($parent->getPriceLevelCountriesIndexedByGroup() as $groupId => $countries) {
				foreach ($countries as $countryId => $row)
					$priceLevelCountries[$groupId][$countryId] = $row->toArray();
			}

			$this->pricesHelper->savePriceLevelCountriesToProduct($vaProd, $priceLevelCountries);
		}

		if (!$va->useGifts && class_exists(EshopGiftsExtension::class)) {
			/** @var ProductGift[] $vaGifts */
			$vaGifts = [];
			/** @var ProductGift[] $prodGifts */
			$prodGifts = [];
			foreach ($em->getRepository(ProductGift::class)->createQueryBuilder('g')
				         ->where('g.product = :id')->setParameter('id', $vaProd->getId())
				         ->getQuery()->getResult() as $row)
				$vaGifts[$row->getId()] = $row;
			foreach ($em->getRepository(ProductGift::class)->createQueryBuilder('g')
				         ->where('g.product = :id')->setParameter('id', $parent->getId())
				         ->getQuery()->getResult() as $row)
				$prodGifts[$row->getId()] = $row;

			foreach ($prodGifts as $k => $v) {
				$g = $vaGifts[$k] ?? new ProductGift($v->getGift(), $vaProd);

				$g->dateFrom = $v->dateFrom;
				$g->dateTo   = $v->dateTo;
				$g->setActive($v->isActive());

				$em->persist($g);
			}
		}

		if (!$va->useLabels) {
			/** @var ProductTag[] $vaTags */
			$vaTags    = $vaProd->getProductTags()->toArray();
			$forRemove = $vaProd->getProductTags()->toArray();
			/** @var ProductTag[] $prodTags */
			$prodTags = [];
			foreach ($this->em->getRepository(ProductTag::class)->createQueryBuilder('pt')
				         ->where('pt.product = :product')->setParameter('product', $parent->getId())
				         ->getQuery()->getResult() as $tagRow)
				$prodTags[$tagRow->getTag()->getId()] = $tagRow;

			foreach ($prodTags as $k => $v) {
				$g            = $vaTags[$k] ?? new ProductTag($vaProd, $v->getTag());
				$g->validFrom = $v->validFrom;
				$g->validTo   = $v->validTo;
				unset($forRemove[$k]);

				$em->persist($g);
				if (!isset($vaTags[$k]))
					$vaProd->addProductTag($g);
			}

			foreach ($forRemove as $v) {
				$vaProd->removeProductTag($v);
				$em->remove($v);
			}
		}

		if (Config::load('enableCountryPrices')) {
			$prices = [];
			foreach ($parent->prices->toArray() as $countryId => $row) {
				/** @var ProductPrice $row */
				$prices[$countryId] = $row->toArray();
				if ($va->usePrice)
					unset($prices[$countryId]['price']);
				if ($va->useRetailPrice)
					unset($prices[$countryId]['retailPrice']);
			}

			$this->pricesHelper->savePricesToProduct($vaProd, $prices);
		}

		if (!$va->usePrice)
			$vaProd->price = $parent->price;
		if (!$va->useRetailPrice)
			$vaProd->retailPrice = $parent->retailPrice;
		if (!$va->useUnlimitedQuantity)
			$vaProd->unlimitedQuantity = $parent->unlimitedQuantity;
		if (!$va->useDiscountDisabled)
			$vaProd->discountDisabled = $parent->discountDisabled;
		if (!$va->useCategoryGiftsAllowed)
			$vaProd->setCategoryGiftsAllowed($parent->isCategoryGiftsAllowed());

		if (!$va->useImportPrice)
			$vaProd->setMoreDataValue('stopImportPrice', $parent->getMoreDataValue('stopImportPrice'));
		$vaProd->setMoreDataValue('stopImportQuantity', $parent->getMoreDataValue('stopImportQuantity'));

		$vaProd->setManufacturer($parent->getManufacturer());
		$vaProd->setVateRate($parent->getVateRate());

		$em->persist($vaProd);
		$em->flush($vaProd);

		$em->persist($va);
		$em->flush($va);
	}
}
