<?php declare(strict_types = 1);

namespace EshopCatalog\Model\TemplateTextType;

use Core\Model\Sites;
use Core\Model\UI\Form\BaseContainer;
use Core\Model\TemplateReader\Providers\ITemplateTextType;
use Core\Model\TemplateReader\Providers\TemplateTextType;
use Doctrine\ORM\Query\Expr\Join;
use Doctrine\ORM\QueryBuilder;
use EshopCatalog\FrontModule\Model\CacheService;
use EshopCatalog\FrontModule\Model\ProductQuery;
use EshopCatalog\FrontModule\Model\ProductsFacade;
use EshopCatalog\FrontModule\Model\Tags;
use EshopCatalog\Model\Entities\Availability;
use EshopCatalog\Model\Entities\Product;
use Core\Model\Entities\EntityManagerDecorator;
use Nette\Caching\Cache;
use Nette\Localization\Translator;

class ProductsWithTag extends TemplateTextType implements ITemplateTextType
{
	/** @var string */
	protected $type = 'eshopCatalogProductsWithTag';

	protected ProductsFacade $productsFacade;

	protected EntityManagerDecorator $em;

	protected Translator $translator;

	protected Sites $sitesService;

	protected Tags $tagsService;

	protected CacheService $cacheService;

	public function __construct(ProductsFacade $productsFacade, EntityManagerDecorator $entityManager, Translator $translator,
	                            Sites $sites, Tags $tags, CacheService $cacheService)
	{
		$this->productsFacade = $productsFacade;
		$this->em             = $entityManager;
		$this->translator     = $translator;
		$this->sitesService   = $sites;
		$this->tagsService    = $tags;
		$this->cacheService   = $cacheService;
	}

	/**
	 * @param BaseContainer $formContainer
	 * @param array         $params
	 * @param array         $texts
	 */
	public function loadContainer($formContainer, $params = [], $texts = [])
	{
		$container = $formContainer->addContainer($params['name']);

		if (!isset($params['tag'])) {
			$opts = ['' => ''];
			foreach ($this->tagsService->getAll() as $tag) {
				$opts[$tag->type] = $tag->name;
			}
			$container->addSelect('tag', 'eshopCatalog.templateTextFields.tagId', $opts);
		}

		if (!isset($params['limit'])) {
			$container->addText('limit', 'eshopCatalog.templateTextFields.limit')
				->setHtmlType('number')
				->setValue(6);
		}

		if (!isset($params['onlyInStock'])) {
			$container->addBool('onlyInStock', 'eshopCatalog.templateTextFields.onlyInStock');
		}

		if (!isset($params['onlyCanAddToCart'])) {
			$container->addBool('onlyCanAddToCart', 'eshopCatalog.templateTextFields.onlyCanAddToCart');
		}
	}

	public function render($params)
	{
		$default                    = $this->getDefault();
		$params['tag']              = $params['tag'] ?: $default['tag'] ?? null;
		$params['limit']            = $params['limit'] ?: $default['limit'] ?? null;
		$params['onlyInStock']      = $params['onlyInStock'] ?: $default['onlyInStock'] ?? false;
		$params['onlyCanAddToCart'] = $params['onlyCanAddToCart'] ?: $default['onlyCanAddToCart'] ?? false;

		if (!isset($params['toVar'])) {
			bdump('Need variable for output');

			return [];
		}
		if (!isset($params['tag'])) {
			bdump('Need tag for output');

			return [];
		}
		$params['tag'] = $params['tag'] ?? '';

		$data = $this->cacheService->defaultCache->load('byTag_' . $params['tag'] . '_ ' . $this->sitesService->getCurrentSite()->getIdent() . '_' . $this->translator->getLocale(), function(&$dep) use ($params) {
			$dep = [
				Cache::TAGS   => ['productsByTag'],
				Cache::EXPIRE => '1 hour',
			];
			$tag = $this->tagsService->getAll()[$params['tag']] ?? null;
			if (!$tag)
				return [];

			$query = (new ProductQuery($this->translator->getLocale()))
				->withTexts(false)
				->withTag($tag->id)
				->inSite($this->sitesService->getCurrentSite()->getIdent())
				->selectIds();

			$query = $query->getQueryBuilder($this->em->getRepository(Product::class));

			/** @var QueryBuilder $query */
			if ($params['onlyInStock'])
				$query->innerJoin('p.availability', 'av', Join::WITH, 'av.ident = :avIdent')
					->setParameter('avIdent', Availability::IN_STOCK);
			if ($params['onlyCanAddToCart'])
				$query->innerJoin('p.availability', 'av2', Join::WITH, 'av2.canAddToCart = 1');

			return $query->getQuery()
				->getScalarResult();
		});

		shuffle($data);
		$data = array_slice($data, 0, $params['limit']);

		$ids = array_map(fn($r) => $r['id'], $data);

		return $this->productsFacade->getProducts($ids);
	}
}
