<?php declare(strict_types = 1);

namespace EshopCatalog\AdminModule\Components\Availability;

use Core\Model\Exceptions\EntityHasRelations;
use Core\Model\UI\BaseControl;
use Core\Model\UI\DataGrid\BaseDataGrid;
use Core\Model\UI\Form\BaseForm;
use EshopCatalog\AdminModule\Model\AvailabilityService;
use EshopCatalog\Model\Entities\Availability;
use Nette\Utils\Html;

class AvailabilityGrid extends BaseControl
{
	/** @var AvailabilityService */
	protected $availabilityService;

	/** @var IAvailabilityFormFactory */
	protected $availabilityFormFactory;

	public function __construct(AvailabilityService $availabilityService, IAvailabilityFormFactory $availabilityFormFactory)
	{
		$this->availabilityService     = $availabilityService;
		$this->availabilityFormFactory = $availabilityFormFactory;
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * ==================  Handle
	 */

	public function handleAdd()
	{
		$this->template->modalTitle = $this->t('eshopCatalog.title.newAvailability');
		$this->template->modal      = 'form';
		$this->redrawControl('modal');
	}

	public function handleEdit(int $id): void
	{
		$this['form']->id = $id;
		$this['form']->setAvailability($id);
		$this->template->modalTitle = $this->t('eshopCatalog.title.editAvailability',
			['title' => $this['form']->availability->getText()->name]);
		$this->template->modal      = 'form';
		$this->redrawControl('modal');
	}

	public function handleDelete(int $id): void
	{
		$presenter = $this->getPresenter();
		try {
			if ($this->availabilityService->remove($id))
				$presenter->flashMessageSuccess('eshopCatalog.defaultGrid.removed');
			else
				$presenter->flashMessageDanger('eshopCatalog.defaultGrid.removeFailed');
		} catch (EntityHasRelations $e) {
			$presenter->flashMessageDanger($e->getMessage());
		}

		if ($presenter->isAjax()) {
			$this['grid']->reload();
			$presenter->redrawControl('flashes');
		} else
			$presenter->redirect('this');
	}

	/*******************************************************************************************************************
	 * ==================  Components
	 */

	protected function createComponentGrid(): BaseDataGrid
	{
		$grid = $this->createGrid();
		$grid->setDefaultPerPage(50);

		$qb = $this->availabilityService->getEr()->createQueryBuilder('a')
			->addSelect('at')
			->leftJoin('a.texts', 'at', 'WITH', 'at.lang = :lang')
			->setParameters([
				'lang' => $this->translator->getLocale(),
			]);

		$grid->setDataSource($qb);

		// Columns
		$grid->addColumnText('name', 'eshopCatalog.availability.name', 'text.name')->setRenderer(function(Availability $row) {
			return Html::el('span', [
				'class' => 'btn',
				'style' => "color: $row->color; background-color: $row->bgColor; cursor: default;",
			])->setText($row->getText()->name);
		});
		$grid->addColumnNumber('delay', 'eshopCatalog.availability.delay');

		// Actions
		$grid->addAction('edit', '', 'edit!')->setIcon('edit')->setBsType('primary')
			->addClass('ajax');
		$grid->addAction('delete', '', 'delete!')->setIcon('times')->setBsType('danger')
			->addClass('ajax')
			->setConfirm('default.reallyDelete')
			->setRenderCondition(function(Availability $row) { return $row->canDelete(); });

		// Columns prototype

		return $grid;
	}

	protected function createComponentForm(): AvailabilityForm
	{
		$control = $this->availabilityFormFactory->create();
		if ($this->getParameter('id'))
			$control->setAvailability((int) $this->getParameter('id'));

		$control['form']->onSuccessSave[]         = function(BaseForm $form) {
			$this->getPresenter()->redrawControl('flashes');
			$this['grid']->reload();
		};
		$control['form']->onSuccessSaveAndClose[] = function(BaseForm $form) {
			$this->getPresenter()->payload->hideModal = true;
			$this->getPresenter()->redrawControl('flashes');
			$this['grid']->reload();
		};
		$control['form']['saveControl']->closeModalOnCancel();

		return $control;
	}
}
