<?php declare(strict_types = 1);

namespace EshopCatalog\AdminModule\Components\Features;

use Core\Model\Event\CreateFormEvent;
use Core\Model\Event\FormSuccessEvent;
use Core\Model\Event\SetFormDataEvent;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use Nette\InvalidArgumentException;
use Nette\Utils\ArrayHash;
use EshopCatalog\Model\Entities\Feature;
use EshopCatalog\Model\Entities\FeatureTexts;
use EshopCatalog\Model\Entities\FeatureValue;
use EshopCatalog\Model\Entities\FeatureValueTexts;
use EshopCatalog\AdminModule\Model\FeatureValues;

class FeatureValueForm extends BaseControl
{
	/** @var featureValue */
	public $featureValue;

	/** @var featureValueTexts */
	public $featureValueTexts;

	/** @var FeatureValues */
	protected $featureValueServices;

	public function __construct(FeatureValues $featureValues)
	{
		$this->featureValueServices = $featureValues;
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	protected function createComponentForm(): BaseForm
	{
		$form = $this->createForm();

		$features = [null => ''];
		foreach ($this->em->getRepository(Feature::class)->createQueryBuilder('f')->select('f.id, ft.name')
			         ->join(FeatureTexts::class, 'ft', 'WITH', 'f.id = ft.id')
			         ->where('ft.lang = :lang')->setParameter('lang', $this->translator->getLocale())
			         ->getQuery()->getResult() as $g) {
			$features[$g['id']] = $g['name'];
		}

		if (count($features) == 1) {
			return $form;
		}

		$form->addSelect('feature', 'eshopCatalog.featureValueForm.feature', $features)->setRequired();
		$form->addText('value', 'eshopCatalog.featureValueForm.value')->setIsMultilanguage()->setMaxLength(255);

		$this->eventDispatcher->dispatch(
			new CreateFormEvent($form, $this->getPresenter(false) ? $this->template : null),
			FeatureValueForm::class . '::createForm'
		);

		$form->addSubmit('saveAndNew', null ?: 'eshopCatalog.defaultForm.saveAndNew')->getControlPrototype()->addHtml('<i class="fa fa-save"></i>');
		$form->addSaveCancelControl('saveControl');

		$form->onSuccess[] = [$this, 'formSuccess'];

		return $form;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values): bool
	{
		try {
			if ($this->featureValue) {
				$featureValue = $this->featureValue;
			} else {
				$featureValue = new FeatureValue();
				$featureValue->setFeature($this->em->getReference(Feature::class, $values->feature));
			}

			$texts      = $featureValue->getFeatureValueTexts()->toArray();
			$langValues = $form->convertMultilangValuesToArray();

			$this->em->persist($featureValue);
			foreach ($langValues as $lang => $vals) {
				if ($vals['value'] == '') {
					if (isset($texts[$lang])) {
						$this->em->remove($texts[$lang]);
						unset($texts[$lang]);
					}
					continue;
				}

				$text = $texts[$lang] ?? new FeatureValueTexts($featureValue, $lang);
				$text->setName($vals['value']);
				$this->em->persist($text);

				$texts[$lang] = $text;
			}

			$featureValue->setTexts($texts);

			$event                   = new FormSuccessEvent($form, $values,
				$this->getPresenter(false) ? $this->template : null, $this->getPresenter(false) ? $this->getPresenter() : null);
			$event->custom['entity'] = $featureValue;
			$this->eventDispatcher->dispatch($event, FeatureValueForm::class . '::formSuccess');

			$this->em->persist($featureValue);
			$this->em->flush();
			$form->addCustomData('featureValueId', $featureValue->getId());
			$this->getPresenter()->flashMessageSuccess('eshopCatalog.featureValueForm.featureValueSaved');

			return true;
		} catch (\Exception $e) {
			$form->addError($e->getMessage());
		}

		return false;
	}

	public function setFeatureValue($id): void
	{
		$this->featureValue = $this->em->getRepository(FeatureValue::class)->find($id);

		if (!$this->featureValue)
			throw new InvalidArgumentException();

		$form = $this['form'];
		$form->setDefaults([
			'feature' => $this->featureValue->getFeature()->getId(),
		]);

		$mTexts = $form->prepareMultilangTextsForForm(['value' => 'name'], $this->featureValue->getFeatureValueTexts()->toArray());
		$form->setDefaults($mTexts['texts']);
		$this->eventDispatcher->dispatch(new SetFormDataEvent($form, $this->featureValue), FeatureValueForm::class . '::setFeatureValue');
	}
}

