<?php declare(strict_types = 1);

namespace EshopCatalog\AdminModule\Components\Manufacturers;

use Core\Components\Flashes\Flashes;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use Nette\Http\IResponse;
use Nette\Utils\ArrayHash;
use EshopCatalog\Model\Entities\Manufacturer;
use EshopCatalog\Model\Entities\ManufacturerTexts;
use EshopCatalog\AdminModule\Model\Manufacturers;

/**
 * Class ManufacturerForm
 * @package EshopCatalog\AdminModule\Components\Manufacturers
 */
class ManufacturerForm extends BaseControl
{
	/** @var Manufacturer */
	public $manufacturer;

	/** @var Manufacturers */
	protected $manufacturerServices;

	public function __construct(Manufacturers $suppliers)
	{
		$this->manufacturerServices = $manufacturers;
	}

	public function render()
	{
		$this->template->render($this->getTemplateFile());
	}

	protected function createComponentForm()
	{
		$form = $this->createForm();

		$form->addText('name', 'eshopCatalog.manufacturerForm.name')->setRequired()->setMaxLength(255);
		$form->addBool('isPublished', 'eshopCatalog.manufacturerForm.isPublished')->setDefaultValue(0);
		$form->addFilesManager('logo', 'eshopCatalog.manufacturerForm.logo');
		$form->addEditor('desc', 'eshopCatalog.manufacturerForm.desc')->setHeight(300)->setIsMultilanguage();

		$form->addSaveCancelControl('saveControl');

		$form->onSuccess[] = [$this, 'formSuccess'];

		return $form;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values)
	{
		try {
			$manufacturer = $this->manufacturer ?: new Manufacturer();

			$manufacturer->name        = $values->name;
			$manufacturer->isPublished = $values->isPublished;
			$manufacturer->logo        = $values->logo;

			$mTexts = $form->prepareMultilangTextsForEntity(['desc' => 'description'], $values);
			foreach ($mTexts as $lang => $vals) {
				$text = $manufacturer->getText($lang) ?: new ManufacturerTexts($manufacturer, $lang);
				foreach ($vals as $k => $v)
					$text->$k = $v;

				$this->em->persist($text);
			}

			$this->em->persist($manufacturer);
			$this->em->flush();
			$form->addCustomData('manufacturerId', $manufacturer->getId());
			$this->getPresenter()->flashMessageSuccess('eshopCatalog.manufacturerForm.manufacturerSaved');
		} catch (\Exception $e) {
			$form->addError($e->getMessage());

			return false;
		}
	}

	public function setManufacturer($id)
	{
		$this->manufacturer = $this->em->getRepository(Manufacturer::class)->find($id);

		if ($this->manufacturer) {
			$form = $this['form'];
			$form->setDefaults([
				'name'        => $this->manufacturer->name,
				'isPublished' => $this->manufacturer->isPublished,
				'logo'        => $this->manufacturer->logo,
			]);

			$mTexts = $form->prepareMultilangTextsForForm(['desc' => 'description'], $this->manufacturer->getTexts()->toArray());
			$form->setDefaults($mTexts['texts']);
		} else
			$this->getPresenter()->error(null, IResponse::S404_NOT_FOUND);
	}
}
