<?php declare(strict_types=1);

namespace EshopCatalog\AdminModule\Model\FormContainers;

use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\SystemConfig;
use Core\Model\UI\AbstractPresenter;
use Core\Model\UI\Form\BaseContainer;
use Core\Model\UI\Form\BaseForm;
use Core\Model\UI\Form\Controls\FilesManagerInput;
use EshopCatalog\Model\Config;
use EshopCatalog\Model\Entities\Product;
use EshopCatalog\Model\Entities\ProductDocument;
use Nette\Application\IPresenter;
use Nette\ComponentModel\IComponent;
use Nette\Localization\ITranslator;
use Nette\Utils\ArrayHash;
use Nette\Utils\Validators;

class DocumentContainer
{
	/** @var ITranslator */
	protected $translator;

	/** @var EntityManagerDecorator */
	protected $em;

	/**
	 * DocumentContainer constructor.
	 * @param ITranslator $translator
	 * @param EntityManagerDecorator $em
	 */
	public function __construct(ITranslator $translator, EntityManagerDecorator $em)
	{
		$this->translator = $translator;
		$this->em = $em;
	}

	/**
	 * @param bool $multilang
	 * @return BaseContainer
	 */
	public function getContainer($multilang = false): BaseContainer
	{
		$container = new BaseContainer();

		for ($i = 0; $i < Config::load('maxDocumentsToUpload', 20); $i++) {
			$documentContainer = new BaseContainer();

			$documentContainer->addText('fileName')
							  ->setPlaceholder($this->translator->translate('eshopCatalog.documents.fileName'));
			$documentContainer->addFilesManager('filePath', 'eshopCatalog.documents.filePath')
							  ->addSecuredRoot([SystemConfig::load('filesManager.securedRoot')])
							  ->addConditionOn($documentContainer['fileName'], BaseForm::FILLED)
							  ->setRequired();
			$documentContainer['fileName']->addConditionOn($documentContainer['filePath'], BaseForm::FILLED)
										  ->setRequired();
			$documentContainer->addLangsSelect('lang', 'eshopCatalog.documents.lang')
							  ->setPrompt(null);
			$documentContainer->addHidden('removeLink');
			$documentContainer->addHidden('id');

			$documentContainer['lang']->setItems([null => 'eshopCatalog.documents.lang'] + $documentContainer['lang']->getItems());

			$container->addComponent($documentContainer, 'document_' . $i);
		}

		return $container;
	}

	/**
	 * @param BaseContainer $container
	 * @param array         $values
	 */
	public function setDefaults(&$container, $values): void
	{
		$i = 0;
		foreach ($container->getComponents() as $documentContainer) {
			foreach ($documentContainer->getControls() as $c) {
				if (isset($values[$i][$c->getName()])) {
					$c->setDefaultValue($values[$i][$c->getName()]);
				}
			}
			$i++;
		}
	}

	/**
	 * @param array $data
	 */
	public function saveData(ArrayHash $data, Product $product): void
	{
		foreach ($data as $document) {
			if (Validators::isNone($document['fileName'])) {
				continue;
			}

			if (Validators::isNone($document['id'])) {
				$this->em->persist(new ProductDocument($document['lang'], $document['fileName'], $document['filePath'], $product));
			} else {
				foreach ($product->getDocuments() as $doc) {
					if ($doc->getId() === ((int) $document['id'])) {
						$doc->lang = $document['lang'] ?? null;
						$doc->name = $document['fileName'] ?? null;
						$doc->file = $document['filePath'] ?? null;
						$this->em->persist($doc);
						break;
					}
				}
			}
		}
	}

}